# -*- coding: utf-8 -*-
from __future__ import print_function, unicode_literals

import os
import subprocess
import time

import docker.errors
import pytest

from ..constants import FEDORA_MINIMAL_REPOSITORY, FEDORA_MINIMAL_REPOSITORY_TAG, \
    FEDORA_REPOSITORY

from conu.backend.docker.client import get_client
from conu import DockerRunBuilder, Probe, ConuException, DockerBackend, DockerImagePullPolicy

from six import string_types


def test_image():
    """
    Basic tests of interacting with image: inspect, tag, remove
    """
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        assert "Config" in image.inspect()
        assert "Config" in image.get_metadata()
        assert "fedora-minimal:26" in image.get_full_name()
        assert "registry.fedoraproject.org/fedora-minimal:26" == str(image)
        assert "DockerImage(repository=%s, tag=%s)" % (FEDORA_MINIMAL_REPOSITORY,
                                                       FEDORA_MINIMAL_REPOSITORY_TAG) == repr(image)
        assert isinstance(image.get_id(), string_types)
        new_image = image.tag_image(tag="test")
        new_image.rmi(via_name=True)


def test_image_wrong_types():
    with DockerBackend() as backend:
        with pytest.raises(ConuException) as exc:
            backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, DockerImagePullPolicy.NEVER)
            assert "tag" in exc.value.message


def test_container():
    """
    Basic tests of interacting with a container
    """
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        c = image.run_via_binary(
            DockerRunBuilder(command=["cat"], additional_opts=["-i", "-t"])
        )
        try:
            assert "Config" in c.inspect()
            assert "Config" in c.get_metadata()
            assert c.get_id() == str(c)
            assert repr(c)
            assert isinstance(c.get_id(), string_types)
        finally:
            c.delete(force=True)


def test_copy_to(tmpdir):
    content = b"gardener did it"
    p = tmpdir.join("secret")
    p.write(content)

    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        c = image.run_via_binary(
            DockerRunBuilder(command=["cat"], additional_opts=["-i", "-t"])
        )
        try:
            c.copy_to(str(p), "/")
            assert [content] == c.execute(["cat", "/secret"])
        finally:
            c.delete(force=True)


def test_copy_from(tmpdir):
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        c = image.run_via_binary(
            DockerRunBuilder(command=["cat"], additional_opts=["-i", "-t"])
        )
        try:
            c.copy_from("/etc/fedora-release", str(tmpdir))
            with open(os.path.join(str(tmpdir), "fedora-release")) as fd:
                assert fd.read() == "Fedora release 26 (Twenty Six)\n"

            c.copy_from("/etc", str(tmpdir))
            os.path.exists(os.path.join(str(tmpdir), "passwd"))
        finally:
            c.delete(force=True)


def test_container_create_failed():
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        # should raise an exc, there is no such command: waldo; we need to find waldo first
        with pytest.raises(ConuException):
            image.run_via_binary(
                DockerRunBuilder(command=["waldo"])
            )
        c = image.run_via_binary_in_foreground(
            DockerRunBuilder(command=["waldo"])
        )
        c.popen_instance.communicate()
        assert c.popen_instance.returncode > 0


def test_interactive_container():
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        command = ["bash"]
        r = DockerRunBuilder(command=command, additional_opts=["-i"])
        cont = image.run_via_binary_in_foreground(
            r, popen_params={"stdin": subprocess.PIPE, "stdout": subprocess.PIPE})
        try:
            assert "" == cont.logs_unicode()
            assert cont.is_running()
            time.sleep(0.1)
            cont.popen_instance.stdin.write(b"echo palacinky\n")
            cont.popen_instance.stdin.flush()
            time.sleep(0.2)
            assert b"palacinky" in cont.popen_instance.stdout.readline()
        finally:
            cont.delete(force=True)


def test_container_logs():
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        command = ["bash", "-c", "for x in `seq 1 5`; do echo $x; done"]
        r = DockerRunBuilder(command=command)
        cont = image.run_via_binary(r)
        try:
            Probe(timeout=5, fnc=cont.get_status, expected_retval='exited').run()
            assert not cont.is_running()
            assert list(cont.logs()) == [b"1\n", b"2\n", b"3\n", b"4\n", b"5\n"]
            assert cont.logs_unicode() == "1\n2\n3\n4\n5\n"
            assert cont.logs_in_bytes() == b"1\n2\n3\n4\n5\n"
        finally:
            cont.delete(force=True)


def test_http_client():
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_REPOSITORY)
        c = image.run_via_binary(
            DockerRunBuilder(command=["python3", "-m", "http.server", "--bind", "0.0.0.0 8000"])
        )
        try:
            c.wait_for_port(8000)
            assert c.is_running()
            r = c.http_request(port="8000")
            assert "<!DOCTYPE HTML PUBLIC" in r.content.decode("utf-8")
            assert r.ok
            r2 = c.http_request(path="/etc", port="8000")
            assert "<!DOCTYPE HTML PUBLIC" in r2.content.decode("utf-8")
            assert "passwd" in r2.content.decode("utf-8")
            assert r2.ok
        finally:
            c.delete(force=True)


def test_wait_for_status():
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        cmd = DockerRunBuilder(command=['sleep', '2'])
        cont = image.run_via_binary(cmd)

        try:
            start = time.time()
            p = Probe(timeout=6, fnc=cont.get_status, expected_retval='exited')
            p.run()
            end = time.time() - start
            assert end > 2, "Probe should wait till container status is exited"
            assert end < 7, "Probe should end when container status is exited"
        finally:
            cont.delete(force=True)


def test_exit_code():
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        cmd = DockerRunBuilder(command=['sleep', '2'])
        cont = image.run_via_binary(cmd)
        try:
            assert cont.is_running() and cont.exit_code() == 0
            p = Probe(timeout=5, fnc=cont.get_status, expected_retval='exited')
            p.run()
            assert not cont.is_running() and cont.exit_code() == 0
        finally:
            cont.delete(force=True)

        cmd = DockerRunBuilder(command=['bash', '-c', "exit 42"])
        cont = image.run_via_binary(cmd)
        try:
            cont.wait()
            assert cont.exit_code() == 42
        finally:
            cont.delete(force=True)


def test_blocking_execute():
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        cmd = DockerRunBuilder(command=['sleep', 'infinity'])
        cont = image.run_via_binary(cmd)
        cont.execute(["bash", "-c", "exit 0"])
        assert [b"asd"] == cont.execute(["printf", "asd"])
        assert [b"asd\nasd"] == cont.execute(["printf", "asd\nasd"])
        with pytest.raises(ConuException) as ex:
            cont.execute(["bash", "-c", "exit 110"])
            assert "exit code 110" in ex.value.message
            assert "bash" in ex.value.message


def test_nonblocking_execute():
    with DockerBackend() as backend:
        image = backend.ImageClass(FEDORA_MINIMAL_REPOSITORY, tag=FEDORA_MINIMAL_REPOSITORY_TAG)
        cmd = DockerRunBuilder(command=['sleep', 'infinity'])
        cont = image.run_via_binary(cmd)
        stream = cont.execute(["bash", "-c", "exit 0"], blocking=False)
        list(stream)
        gen = cont.execute(["printf", "asd"], blocking=False)
        assert [b"asd"] == list(gen)
        gen = cont.execute(["printf", "asd\nasd"], blocking=False)
        assert [b"asd\nasd"] == list(gen)
        cont.execute(["bash", "-c", "sleep 0.01; exit 110"], blocking=False)


def test_pull_always():
    with DockerBackend() as backend:
        image = backend.ImageClass("docker.io/library/busybox", tag="1.25.1",
                                   pull_policy=DockerImagePullPolicy.ALWAYS)
        try:
            assert image.is_present()
        finally:
            image.rmi(force=True)


def test_pull_if_not_present():
    with DockerBackend() as backend:
        with pytest.raises(docker.errors.DockerException):
            get_client().inspect_image("docker.io/library/busybox:1.25.1")
        image = backend.ImageClass("docker.io/library/busybox", tag="1.25.1")
        try:
            assert image.is_present()
        finally:
            image.rmi(force=True)


def test_pull_never():
    with DockerBackend() as backend:
        with pytest.raises(docker.errors.DockerException):
            get_client().inspect_image("docker.io/library/busybox:1.25.1")
        image = backend.ImageClass("docker.io/library/busybox", tag="1.25.1",
                                   pull_policy=DockerImagePullPolicy.NEVER)
        assert not image.is_present()
