"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mesh = exports.MeshFilterType = void 0;
const cdk = require("@aws-cdk/core");
const appmesh_generated_1 = require("./appmesh.generated");
const virtual_gateway_1 = require("./virtual-gateway");
const virtual_node_1 = require("./virtual-node");
const virtual_router_1 = require("./virtual-router");
const virtual_service_1 = require("./virtual-service");
/**
 * (experimental) A utility enum defined for the egressFilter type property, the default of DROP_ALL, allows traffic only to other resources inside the mesh, or API calls to amazon resources.
 *
 * @default DROP_ALL
 * @experimental
 */
var MeshFilterType;
(function (MeshFilterType) {
    MeshFilterType["ALLOW_ALL"] = "ALLOW_ALL";
    MeshFilterType["DROP_ALL"] = "DROP_ALL";
})(MeshFilterType = exports.MeshFilterType || (exports.MeshFilterType = {}));
/**
 * Represents a new or imported AppMesh mesh
 */
class MeshBase extends cdk.Resource {
    /**
     * (experimental) Adds a VirtualRouter to the Mesh with the given id and props.
     *
     * @experimental
     */
    addVirtualRouter(id, props = {}) {
        return new virtual_router_1.VirtualRouter(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * (experimental) Adds a VirtualService with the given id.
     *
     * @experimental
     */
    addVirtualService(id, props = {}) {
        return new virtual_service_1.VirtualService(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * (experimental) Adds a VirtualNode to the Mesh.
     *
     * @experimental
     */
    addVirtualNode(id, props = {}) {
        return new virtual_node_1.VirtualNode(this, id, {
            ...props,
            mesh: this,
        });
    }
    /**
     * (experimental) Adds a VirtualGateway to the Mesh.
     *
     * @experimental
     */
    addVirtualGateway(id, props) {
        return new virtual_gateway_1.VirtualGateway(this, id, {
            ...props,
            mesh: this,
        });
    }
}
/**
 * (experimental) Define a new AppMesh mesh.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/meshes.html
 * @experimental
 */
class Mesh extends MeshBase {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.meshName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        const mesh = new appmesh_generated_1.CfnMesh(this, 'Resource', {
            meshName: this.physicalName,
            spec: {
                egressFilter: props.egressFilter ? {
                    type: props.egressFilter,
                } : undefined,
            },
        });
        this.meshName = this.getResourceNameAttribute(mesh.attrMeshName);
        this.meshArn = this.getResourceArnAttribute(mesh.ref, {
            service: 'appmesh',
            resource: 'mesh',
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing mesh by arn.
     *
     * @experimental
     */
    static fromMeshArn(scope, id, meshArn) {
        const parts = cdk.Stack.of(scope).parseArn(meshArn);
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = parts.resourceName || '';
                this.meshArn = meshArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Import an existing mesh by name.
     *
     * @experimental
     */
    static fromMeshName(scope, id, meshName) {
        const arn = cdk.Stack.of(scope).formatArn({
            service: 'appmesh',
            resource: 'mesh',
            resourceName: meshName,
        });
        class Import extends MeshBase {
            constructor() {
                super(...arguments);
                this.meshName = meshName;
                this.meshArn = arn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Mesh = Mesh;
//# sourceMappingURL=data:application/json;base64,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