import json
import re
import requests
from urllib.parse import urlparse, urljoin, quote
from requests.packages.urllib3.exceptions import InsecureRequestWarning
from .group import Group
from .person import Person
from .user import User
from .verbose import VERBOSE
from .mailinglist import Mailinglist


class ETH_IAM_conn():
    def __init__(self, admin_username, admin_password, hostname, endpoint_base):
        self._admin_username = admin_username
        self._admin_password = admin_password
        self.hostname = hostname
        self.endpoint_base = endpoint_base
        self.verify_certificates = True
        self.timeout = 600
        self._get_version()

    def _get_version(self, endpoint='/version'):
        full_url = urljoin(self.hostname, self.endpoint_base+endpoint)
        resp = requests.get(
            full_url,
            headers={'Accept': 'application/json'},
            auth=(self._admin_username, self._admin_password),
            verify=self.verify_certificates,
            timeout=self.timeout,
        )
        if resp.ok:
            data = json.loads(resp.content.decode())
            '''
            {"ETH IAM Web services":{"build date":"2021-01-12 10:23","build version":"2019-1.2"}}'
            '''
            self.build_date = data.get('ETH IAM Web services',{}).get('build date')
            self.build_version = data.get('ETH IAM Web services',{}).get('build version')
        else:
            raise ValueError('a general error occured')


    def _delete_request(self, 
            endpoint, 
            success_msg=None,
            not_allowed_msg=None,
            failed_msg=None,
        ):
        full_url = urljoin(self.hostname, self.endpoint_base+endpoint)
        resp = requests.delete(
            full_url,
            headers={'Accept': 'application/json'},
            auth=(self._admin_username, self._admin_password),
            verify=self.verify_certificates,
            timeout=self.timeout,
        )
        if not success_msg:
            return resp

        if resp.ok:
            if VERBOSE:
                print(success_msg)
        elif resp.status_code == 401:
            if not_allowed_msg is None:
                not_allowed_msg =f'You are NOT ALLOWED to do a DELETE operation on {endpoint}'
            raise ValueError(not_allowed_msg)
        else:
            data = json.loads(resp.content.decode())
            if not failed_msg:
                failed_msg = f"FAILED to do a DELETE operation on {endpoint}"
            raise ValueError(failed_msg + ': ' + data['message'])


    def _post_request(self,
            endpoint,
            body,
            success_msg=None,
            not_allowed_msg=None,
            failed_msg=None,
    ):
        full_url = urljoin(self.hostname, self.endpoint_base+endpoint)
        resp = requests.post(
            full_url,
            json.dumps(body),
            headers={
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            auth=(self._admin_username, self._admin_password),
            verify=self.verify_certificates,
            timeout=self.timeout,
        )
        if not success_msg:
            return resp

        if resp.ok:
            if VERBOSE:
                print(success_msg)
        elif resp.status_code == 401:
            if not_allowed_msg is None:
                not_allowed_msg =f'You are NOT ALLOWED to do a POST operation on {endpoint}'
            raise ValueError(not_allowed_msg)
        else:
            data = json.loads(resp.content.decode())
            if not failed_msg:
                failed_msg = f"FAILED to do a POST operation on {endpoint}"
            raise ValueError(failed_msg + ': ' + data['message'])

    def _put_request(self,
            endpoint,
            body,
            success_msg=None,
            not_allowed_msg=None,
            failed_msg=None,
    ):
        full_url = urljoin(self.hostname, self.endpoint_base+endpoint)
        resp = requests.put(
            full_url,
            json.dumps(body),
            headers={
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            },
            auth=(self._admin_username, self._admin_password),
            verify=self.verify_certificates,
            timeout=self.timeout,
        )
        if not success_msg:
            return resp

        if resp.ok:
            if VERBOSE:
                print(success_msg)
        elif resp.status_code == 401:
            if not_allowed_msg is None:
                not_allowed_msg =f'You are NOT ALLOWED to do a PUT operation on {endpoint}'
            raise ValueError(not_allowed_msg)
        else:
            data = json.loads(resp.content.decode())
            if not failed_msg:
                failed_msg = f"FAILED to do a PUT operation on {endpoint}"
            raise ValueError(failed_msg + ': ' + data['message'])

    def _get_request(self, endpoint):
        full_url = urljoin(self.hostname, self.endpoint_base+endpoint)
        resp = requests.get(
            full_url,
            headers={'Accept': 'application/json'},
            auth=(self._admin_username, self._admin_password),
            verify=self.verify_certificates,
            timeout=self.timeout,
        )
        return resp

    def new_person(self, firstname, lastname):
        raise Exception("not implemented yet")
        return Person(conn=self, firstname=firstname, lastname=lastname)

    def get_person(self, identifier=None, **kwargs):
        if identifier is not None:
            endpoint = '/usermgr/person/{}'.format(identifier)
        elif kwargs:
            args = "&".join("{}={}".format(key, val)
                            for key, val in kwargs.items())
            endpoint = '/usermgr/person?{}'.format(args)
        else:
            raise ValueError("please provide an identifier")

        resp = self._get_request(endpoint)
        data = json.loads(resp.content.decode())
        if resp.ok:
            return Person(conn=self, data=data)
        elif resp.status_code == 401:
            raise ValueError('Provided admin-username/password is incorrect or you are not allowed to do this operation')
        else:
            raise ValueError(data['message'])

    def get_user(self, identifier):
        endpoint = '/usermgr/user/{}'.format(identifier)
        resp = self._get_request(endpoint)
        data = json.loads(resp.content.decode())
        if resp.ok:
            return User(conn=self, data=data)
        elif resp.status_code == 401:
            raise ValueError('Provided admin-username/password is incorrect or you are not allowed to do this operation')
        else:
            raise ValueError(data['message'])

    def new_group(self, name, description, admingroup, targets, members=None):
        """
        name=<Group Name>
        description=<what is the purpose of this group>
        admingroup=<Admin Group>
        targets=['AD', 'LDAPS'] -- specify at least one target system
        members=['username1', 'username2']
        """
        if members is None:
            members = []

        endpoint = '/groupmgr/group'
        body = {
            "name": name,
            "description": description,
            "admingroup": admingroup,
            "targets": targets,
            "members": members
        }
        resp = self._post_request(endpoint, body)
        if resp.ok:
            data = json.loads(resp.content.decode())
            if VERBOSE:
                print("new group {} was successfully created".format(name))
            return Group(conn=self, data=data)
        elif resp.status_code == 401:
            raise ValueError('Provided admin-username/password is incorrect or you are not allowed to do this operation')
        else:
            data = json.loads(resp.content.decode())
            raise ValueError(data['message'])

    def del_group(self, name):
        """Deletes a group and removes it from all its target systems.
        """
        endpoint = '/groupmgr/group/{}'.format(name)
        resp = self._delete_request(endpoint)
        if resp.ok:
            if VERBOSE:
                print("group {} was successfully deleted".format(name))
        elif resp.status_code == 401:
            raise ValueError('Provided admin-username/password is incorrect or you are not allowed to do this operation')
        else:
            data = json.loads(resp.content.decode())
            raise ValueError(data['message'])

    def get_groups(self, **kwargs):
        """
        agroup=<Admin Group>  -- Get all groups of a given admin group
        name=group_name*      -- all groups starting with «group_name*»
        """
        if kwargs:
            args = "&".join("{}={}".format(key, val)
                            for key, val in kwargs.items())
            endpoint = '/groupmgr/groups?{}'.format(args)
        else:
            raise ValueError(
                "please provide a name or agroup parameter (or both)")

        resp = self._get_request(endpoint)
        data = json.loads(resp.content.decode())
        if resp.ok:
            groups = []
            for item in data:
                groups.append(Group(conn=self, data=item))

            return groups
        elif resp.status_code == 401:
            raise ValueError('Provided admin-username/password is incorrect or you are not allowed to do this operation')
        else:
            raise ValueError(data['message'])

    def get_group(self, identifier=None):
        if identifier is not None:

            if re.search(r'^\d+$', str(identifier)):
                # we searched for a gidNumber
                groups = self.get_groups(gidNumber=identifier)
                if len(groups) == 1:
                    return groups[0]
                else:
                    raise ValueError(
                        'No group found with gidNumber={}'.format(identifier))
            else:
                endpoint = '/groupmgr/group/{}'.format(identifier)
        else:
            raise ValueError("please provide an identifier")
        resp = self._get_request(endpoint)
        data = json.loads(resp.content.decode())
        if resp.ok:
            return Group(conn=self, data=data)
        elif resp.status_code == 401:
            raise ValueError('Provided admin-username/password is incorrect or you are not allowed to do this operation')
        else:
            raise ValueError(data['message'])

    def get_mailinglist(self, identifier=None, **kwargs):
        if identifier is not None:
            endpoint = '/mailinglists/{}'.format(identifier)
        elif kwargs:
            args = "&".join("{}={}".format(key, val)
                            for key, val in kwargs.items())
            endpoint = '/mailinglists/?{}'.format(args)
        else:
            raise ValueError("please provide an identifier")
        resp = self._get_request(endpoint)
        data = json.loads(resp.content.decode())
        if resp.ok:
            return Mailinglist(conn=self, data=data)
        elif resp.status_code == 401:
            raise ValueError('Provided admin-username/password is incorrect or you are not allowed to do this operation')
        else:
            raise ValueError(data['message'])
