"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LaunchTemplateRequireImdsv2Aspect = exports.InstanceRequireImdsv2Aspect = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const ec2_generated_1 = require("../ec2.generated");
const instance_1 = require("../instance");
const launch_template_1 = require("../launch-template");
/**
 * Base class for Aspect that makes IMDSv2 required.
 */
class RequireImdsv2Aspect {
    constructor(props) {
        this.suppressWarnings = props?.suppressWarnings ?? false;
    }
    /**
     * Adds a warning annotation to a node, unless `suppressWarnings` is true.
     *
     * @param node The scope to add the warning to.
     * @param message The warning message.
     */
    warn(node, message) {
        if (this.suppressWarnings !== true) {
            cdk.Annotations.of(node).addWarning(`${RequireImdsv2Aspect.name} failed on node ${node.node.id}: ${message}`);
        }
    }
}
/**
 * Aspect that applies IMDS configuration on EC2 Instance constructs.
 *
 * This aspect configures IMDS on an EC2 instance by creating a Launch Template with the
 * IMDS configuration and associating that Launch Template with the instance. If an Instance
 * is already associated with a Launch Template, a warning will (optionally) be added to the
 * construct node and it will be skipped.
 *
 * To cover Instances already associated with Launch Templates, use `LaunchTemplateImdsAspect`.
 */
class InstanceRequireImdsv2Aspect extends RequireImdsv2Aspect {
    constructor(props) {
        super(props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_InstanceRequireImdsv2AspectProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, InstanceRequireImdsv2Aspect);
            }
            throw error;
        }
        this.suppressLaunchTemplateWarning = props?.suppressLaunchTemplateWarning ?? false;
    }
    visit(node) {
        if (!(node instanceof instance_1.Instance)) {
            return;
        }
        if (node.instance.launchTemplate !== undefined) {
            this.warn(node, 'Cannot toggle IMDSv1 because this Instance is associated with an existing Launch Template.');
            return;
        }
        const launchTemplate = new ec2_generated_1.CfnLaunchTemplate(node, 'LaunchTemplate', {
            launchTemplateData: {
                metadataOptions: {
                    httpTokens: 'required',
                },
            },
        });
        if (core_1.FeatureFlags.of(node).isEnabled(cxapi.EC2_UNIQUE_IMDSV2_LAUNCH_TEMPLATE_NAME)) {
            launchTemplate.launchTemplateName = cdk.Names.uniqueId(launchTemplate);
        }
        else {
            launchTemplate.launchTemplateName = `${node.node.id}LaunchTemplate`;
        }
        node.instance.launchTemplate = {
            launchTemplateName: launchTemplate.launchTemplateName,
            version: launchTemplate.getAtt('LatestVersionNumber').toString(),
        };
    }
    warn(node, message) {
        if (this.suppressLaunchTemplateWarning !== true) {
            super.warn(node, message);
        }
    }
}
exports.InstanceRequireImdsv2Aspect = InstanceRequireImdsv2Aspect;
_a = JSII_RTTI_SYMBOL_1;
InstanceRequireImdsv2Aspect[_a] = { fqn: "@aws-cdk/aws-ec2.InstanceRequireImdsv2Aspect", version: "1.198.0" };
/**
 * Aspect that applies IMDS configuration on EC2 Launch Template constructs.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-launchtemplate-launchtemplatedata-metadataoptions.html
 */
class LaunchTemplateRequireImdsv2Aspect extends RequireImdsv2Aspect {
    constructor(props) {
        super(props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_ec2_LaunchTemplateRequireImdsv2AspectProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, LaunchTemplateRequireImdsv2Aspect);
            }
            throw error;
        }
    }
    visit(node) {
        if (!(node instanceof launch_template_1.LaunchTemplate)) {
            return;
        }
        const launchTemplate = node.node.tryFindChild('Resource');
        const data = launchTemplate.launchTemplateData;
        if (cdk.isResolvableObject(data)) {
            this.warn(node, 'LaunchTemplateData is a CDK token.');
            return;
        }
        const metadataOptions = data.metadataOptions;
        if (cdk.isResolvableObject(metadataOptions)) {
            this.warn(node, 'LaunchTemplateData.MetadataOptions is a CDK token.');
            return;
        }
        const newData = {
            ...data,
            metadataOptions: {
                ...metadataOptions,
                httpTokens: 'required',
            },
        };
        launchTemplate.launchTemplateData = newData;
    }
}
exports.LaunchTemplateRequireImdsv2Aspect = LaunchTemplateRequireImdsv2Aspect;
_b = JSII_RTTI_SYMBOL_1;
LaunchTemplateRequireImdsv2Aspect[_b] = { fqn: "@aws-cdk/aws-ec2.LaunchTemplateRequireImdsv2Aspect", version: "1.198.0" };
//# sourceMappingURL=data:application/json;base64,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