#
# This file generates the sidebar/toctree for all RobotPy projects and should
# be copied to each project when it is updated
#

import os
import os.path
import toml

from .utils import write_if_changed
import urllib.request


def generate_sidebar(
    conf, this_project, url, out="_sidebar.inc.rst", cfg="_sidebar.toml", lang="en"
):
    """
    Call this from your sphinx project's conf.py::

        generate_sidebar(globals(), "name", "url")

    Your toplevel index.rst should have something like this:

        .. include:: _sidebar.rst.inc
    """

    # determine 'latest' or 'stable'
    force_dl = conf["on_rtd"]
    do_gen = os.environ.get("SIDEBAR", None) == "1" or conf["on_rtd"]
    version = conf["rtd_version"]

    lines = ["", ".. DO NOT MODIFY! THIS PAGE IS AUTOGENERATED!", ""]

    def toctree(name):
        lines.extend([".. toctree::", f"    :caption: {name}", "    :maxdepth: 2", ""])

    def endl():
        lines.append("")

    def write(project, desc, link):
        if project == this_project:
            args = desc, link
        elif not do_gen:
            return
        else:
            project_url = data["projects"][project]
            args = (
                desc,
                f"{project_url}/{lang}/{version}/{link}.html",
            )

        lines.append("    %s <%s>" % args)

    if force_dl or not os.path.exists(cfg):
        urllib.request.urlretrieve(url, cfg)

    with open(cfg) as fp:
        data = toml.load(fp)

    for tt in data["toctree"]:
        toctree(tt["name"])

        for item in tt["items"]:
            write(item["p"], item["k"], item["v"])

        endl()

    write_if_changed("_sidebar.rst.inc", "\n".join(lines))
