#!/usr/bin/env python3
"""
    A custom documentation generator because sphinx-apidoc doesn't quite do
    what I want it to. The autosummary thing is really close to what I want.
    However, the documentation for how to use it is lacking.
    
    This is designed to generate docs for our flat import structure in a
    manner that is friendly to the readthedocs theme.
"""

import fnmatch
import inspect
import os

from os.path import abspath, join, exists
from .utils import write_if_changed

from sphinx_automodapi.utils import find_mod_objs

mod_doc = """
%(header)s

.. module:: %(module)s

%(include)s%(fnlink)s

.. autosummary::
    %(names)s

.. toctree::
    :hidden:

    %(files)s

"""

cls_doc = """
%(header)s

.. autoclass:: %(package_name)s.%(cls_name)s
    :members:
    :undoc-members:
    :show-inheritance:
"""


def _heading(name, c):
    return "%s\n%s" % (name, c * len(name))


def gen_package(root: str, package_name: str, include=None, exclude=None):
    """
    Writes rst files describing a package
    """

    if not exclude:
        exclude = []
    if include is None:
        include = ["*"]

    found = find_mod_objs(package_name)

    docdir = abspath(join(root, package_name))
    fnrst = abspath(join(root, package_name, "functions.rst"))

    pkgbase = f"{package_name}.rst"
    pkgrst = abspath(join(root, pkgbase))
    incbase = f"{pkgbase}.inc"
    incrst = abspath(join(root, incbase))

    old_files = {}
    if exists(docdir):
        for fname in os.listdir(docdir):
            old_files[join(docdir, fname)] = True
    else:
        os.mkdir(docdir)

    classes = []
    functions = []

    for name, _, obj in zip(*found):

        included = False
        for pattern in include:
            if fnmatch.fnmatch(name, pattern):
                included = True
                break

        if not included:
            continue

        excluded = False
        for pattern in exclude:
            if fnmatch.fnmatch(name, pattern):
                excluded = True
                break

        if excluded:
            continue

        if inspect.isclass(obj):
            fname = join(docdir, f"{name}.rst")
            write_if_changed(
                fname,
                cls_doc
                % {
                    "header": _heading(name, "-"),
                    "package_name": package_name,
                    "cls_name": name,
                },
            )

            old_files.pop(fname, None)
            classes.append(name)

        elif inspect.isroutine(obj):
            functions.append(name)

    classes = sorted(classes)
    functions = sorted(functions)

    # Create toctree

    files = ["%s/%s" % (package_name, clsname) for clsname in classes]
    fnlink = ""

    if functions:
        files.insert(0, f"{package_name}/functions")
        fnlink = f":doc:`{package_name} functions <{package_name}/functions>`"

        functions_doc = _heading(f"{package_name} functions", "-").split("\n")
        functions_doc.append(f".. currentmodule:: {package_name}")
        for fn in functions:
            functions_doc += [
                "",
                f".. autofunction:: {package_name}.{fn}",
            ]

        write_if_changed(fnrst, "\n".join(functions_doc))
        old_files.pop(fnrst, None)

    include = ""
    if exists(incrst):
        include = f".. include:: {incbase}\n\n"

    write_if_changed(
        pkgrst,
        mod_doc
        % {
            "header": _heading(package_name + " Package", "="),
            "fnlink": fnlink,
            "include": include,
            "module": package_name,
            "names": "\n    ".join(classes),
            "files": "\n    ".join(files),
        },
    )

    old_files.pop(pkgrst, None)

    # delete any files that were not written, since this is an autogenerated directory
    for fname in old_files.keys():
        os.unlink(fname)
