"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const index_1 = require("../lib/index");
const sfn = require("@aws-cdk/aws-stepfunctions");
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const aws_s3_1 = require("@aws-cdk/aws-s3");
function deployNewStateMachine(stack) {
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        }
    };
    return new index_1.S3ToStepFunction(stack, 'test-s3-step-function', props);
}
test('snapshot test S3ToStepFunction default params', () => {
    const stack = new cdk.Stack();
    deployNewStateMachine(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check deployCloudTrail = false', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        deployCloudTrail: false
    };
    const construct = new index_1.S3ToStepFunction(stack, 'test-s3-step-function', props);
    expect(construct.cloudtrail === undefined);
});
test('override eventRuleProps', () => {
    const stack = new cdk.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket');
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        existingBucketObj: mybucket,
        eventRuleProps: {
            eventPattern: {
                source: ['aws.s3'],
                detailType: ['AWS API Call via CloudTrail'],
                detail: {
                    eventSource: [
                        "s3.amazonaws.com"
                    ],
                    eventName: [
                        "GetObject"
                    ],
                    requestParameters: {
                        bucketName: [
                            mybucket.bucketName
                        ]
                    }
                }
            }
        }
    };
    new index_1.S3ToStepFunction(stack, 'test-s3-step-function', props);
    expect(stack).toHaveResource('AWS::Events::Rule', {
        EventPattern: {
            "source": [
                "aws.s3"
            ],
            "detail-type": [
                "AWS API Call via CloudTrail"
            ],
            "detail": {
                eventSource: [
                    "s3.amazonaws.com"
                ],
                eventName: [
                    "GetObject"
                ],
                requestParameters: {
                    bucketName: [
                        {
                            Ref: "mybucket160F8132"
                        }
                    ]
                }
            }
        },
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "tests3stepfunctiontests3stepfunctioneventrulestepfunctionconstructStateMachineA4D4F2B0"
                },
                Id: "Target0",
                RoleArn: {
                    "Fn::GetAtt": [
                        "tests3stepfunctiontests3stepfunctioneventrulestepfunctionconstructEventsRuleRole07FD7EC1",
                        "Arn"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStateMachine(stack);
    expect(construct.cloudtrail !== null);
    expect(construct.stateMachine !== null);
    expect(construct.s3Bucket !== null);
    expect(construct.cloudwatchAlarms !== null);
    expect(construct.stateMachineLogGroup !== null);
    expect(construct.s3LoggingBucket !== null);
    expect(construct.cloudtrail !== null);
    expect(construct.cloudtrailBucket !== null);
    expect(construct.cloudtrailLoggingBucket !== null);
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new cdk.Stack();
    const testBucket = new aws_s3_1.Bucket(stack, 'test-bucket', {});
    const startState = new sfn.Pass(stack, 'StartState');
    const app = () => {
        // Helper declaration
        new index_1.S3ToStepFunction(stack, "bad-s3-args", {
            stateMachineProps: {
                definition: startState
            },
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: cdk.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiczMtc3RlcC1mdW5jdGlvbi50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiczMtc3RlcC1mdW5jdGlvbi50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCw0Q0FBNkM7QUFDN0Msd0NBQXVFO0FBQ3ZFLGtEQUFrRDtBQUNsRCxnQ0FBOEI7QUFDOUIscUNBQXFDO0FBQ3JDLDRDQUF5QztBQUV6QyxTQUFTLHFCQUFxQixDQUFDLEtBQWdCO0lBRTdDLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQUM7SUFFckQsTUFBTSxLQUFLLEdBQTBCO1FBQ25DLGlCQUFpQixFQUFFO1lBQ2pCLFVBQVUsRUFBRSxVQUFVO1NBQ3ZCO0tBQ0YsQ0FBQztJQUVGLE9BQU8sSUFBSSx3QkFBZ0IsQ0FBQyxLQUFLLEVBQUUsdUJBQXVCLEVBQUUsS0FBSyxDQUFDLENBQUM7QUFDckUsQ0FBQztBQUVELElBQUksQ0FBQywrQ0FBK0MsRUFBRSxHQUFHLEVBQUU7SUFDekQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIscUJBQXFCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDN0IsTUFBTSxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUMvRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnQ0FBZ0MsRUFBRSxHQUFHLEVBQUU7SUFDMUMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztJQUVyRCxNQUFNLEtBQUssR0FBMEI7UUFDbkMsaUJBQWlCLEVBQUU7WUFDakIsVUFBVSxFQUFFLFVBQVU7U0FDdkI7UUFDRCxnQkFBZ0IsRUFBRSxLQUFLO0tBQ3hCLENBQUM7SUFFRixNQUFNLFNBQVMsR0FBRyxJQUFJLHdCQUFnQixDQUFDLEtBQUssRUFBRSx1QkFBdUIsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUU5RSxNQUFNLENBQUMsU0FBUyxDQUFDLFVBQVUsS0FBSyxTQUFTLENBQUMsQ0FBQztBQUM3QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx5QkFBeUIsRUFBRSxHQUFHLEVBQUU7SUFDbkMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFOUIsTUFBTSxRQUFRLEdBQUcsSUFBSSxlQUFNLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO0lBQy9DLE1BQU0sVUFBVSxHQUFHLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsWUFBWSxDQUFDLENBQUM7SUFFckQsTUFBTSxLQUFLLEdBQTBCO1FBQ25DLGlCQUFpQixFQUFFO1lBQ2pCLFVBQVUsRUFBRSxVQUFVO1NBQ3ZCO1FBQ0QsaUJBQWlCLEVBQUUsUUFBUTtRQUMzQixjQUFjLEVBQUU7WUFDZCxZQUFZLEVBQUU7Z0JBQ1osTUFBTSxFQUFFLENBQUMsUUFBUSxDQUFDO2dCQUNsQixVQUFVLEVBQUUsQ0FBQyw2QkFBNkIsQ0FBQztnQkFDM0MsTUFBTSxFQUFFO29CQUNOLFdBQVcsRUFBRTt3QkFDWCxrQkFBa0I7cUJBQ25CO29CQUNELFNBQVMsRUFBRTt3QkFDVCxXQUFXO3FCQUNaO29CQUNELGlCQUFpQixFQUFFO3dCQUNqQixVQUFVLEVBQUU7NEJBQ1YsUUFBUSxDQUFDLFVBQVU7eUJBQ3BCO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUM7SUFFRixJQUFJLHdCQUFnQixDQUFDLEtBQUssRUFBRSx1QkFBdUIsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUU1RCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLG1CQUFtQixFQUFFO1FBQ2hELFlBQVksRUFBRTtZQUNaLFFBQVEsRUFBRTtnQkFDUixRQUFRO2FBQ1Q7WUFDRCxhQUFhLEVBQUU7Z0JBQ2IsNkJBQTZCO2FBQzlCO1lBQ0QsUUFBUSxFQUFFO2dCQUNSLFdBQVcsRUFBRTtvQkFDWCxrQkFBa0I7aUJBQ25CO2dCQUNELFNBQVMsRUFBRTtvQkFDVCxXQUFXO2lCQUNaO2dCQUNELGlCQUFpQixFQUFFO29CQUNqQixVQUFVLEVBQUU7d0JBQ1Y7NEJBQ0UsR0FBRyxFQUFFLGtCQUFrQjt5QkFDeEI7cUJBQ0Y7aUJBQ0Y7YUFDRjtTQUNGO1FBQ0QsS0FBSyxFQUFFLFNBQVM7UUFDaEIsT0FBTyxFQUFFO1lBQ1A7Z0JBQ0UsR0FBRyxFQUFFO29CQUNILEdBQUcsRUFBRSx3RkFBd0Y7aUJBQzlGO2dCQUNELEVBQUUsRUFBRSxTQUFTO2dCQUNiLE9BQU8sRUFBRTtvQkFDUCxZQUFZLEVBQUU7d0JBQ1osMEZBQTBGO3dCQUMxRixLQUFLO3FCQUNOO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGtCQUFrQixFQUFFLEdBQUcsRUFBRTtJQUM1QixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUU5QixNQUFNLFNBQVMsR0FBcUIscUJBQXFCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFakUsTUFBTSxDQUFDLFNBQVMsQ0FBQyxVQUFVLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDdEMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxZQUFZLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDeEMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxRQUFRLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDcEMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsS0FBSyxJQUFJLENBQUMsQ0FBQztJQUM1QyxNQUFNLENBQUMsU0FBUyxDQUFDLG9CQUFvQixLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ2hELE1BQU0sQ0FBQyxTQUFTLENBQUMsZUFBZSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQzNDLE1BQU0sQ0FBQyxTQUFTLENBQUMsVUFBVSxLQUFLLElBQUksQ0FBQyxDQUFDO0lBQ3RDLE1BQU0sQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDNUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyx1QkFBdUIsS0FBSyxJQUFJLENBQUMsQ0FBQztBQUNyRCxDQUFDLENBQUMsQ0FBQztBQUVILGlFQUFpRTtBQUNqRSxvREFBb0Q7QUFDcEQsaUVBQWlFO0FBQ2pFLElBQUksQ0FBQyxtREFBbUQsRUFBRSxHQUFHLEVBQUU7SUFDN0QsUUFBUTtJQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBRTlCLE1BQU0sVUFBVSxHQUFHLElBQUksZUFBTSxDQUFDLEtBQUssRUFBRSxhQUFhLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDeEQsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztJQUVyRCxNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixxQkFBcUI7UUFDckIsSUFBSSx3QkFBZ0IsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO1lBQ3pDLGlCQUFpQixFQUFFO2dCQUNqQixVQUFVLEVBQUUsVUFBVTthQUN2QjtZQUNELGlCQUFpQixFQUFFLFVBQVU7WUFDN0IsV0FBVyxFQUFFO2dCQUNYLGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU87YUFDekM7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUM7SUFDRixZQUFZO0lBQ1osTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksRUFBRSxDQUFDO0FBQzdCLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IDIwMjEgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgeyBTeW50aFV0aWxzIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCB7IFMzVG9TdGVwRnVuY3Rpb24sIFMzVG9TdGVwRnVuY3Rpb25Qcm9wcyB9IGZyb20gJy4uL2xpYi9pbmRleCc7XG5pbXBvcnQgKiBhcyBzZm4gZnJvbSAnQGF3cy1jZGsvYXdzLXN0ZXBmdW5jdGlvbnMnO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBCdWNrZXQgfSBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuXG5mdW5jdGlvbiBkZXBsb3lOZXdTdGF0ZU1hY2hpbmUoc3RhY2s6IGNkay5TdGFjaykge1xuXG4gIGNvbnN0IHN0YXJ0U3RhdGUgPSBuZXcgc2ZuLlBhc3Moc3RhY2ssICdTdGFydFN0YXRlJyk7XG5cbiAgY29uc3QgcHJvcHM6IFMzVG9TdGVwRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBzdGF0ZU1hY2hpbmVQcm9wczoge1xuICAgICAgZGVmaW5pdGlvbjogc3RhcnRTdGF0ZVxuICAgIH1cbiAgfTtcblxuICByZXR1cm4gbmV3IFMzVG9TdGVwRnVuY3Rpb24oc3RhY2ssICd0ZXN0LXMzLXN0ZXAtZnVuY3Rpb24nLCBwcm9wcyk7XG59XG5cbnRlc3QoJ3NuYXBzaG90IHRlc3QgUzNUb1N0ZXBGdW5jdGlvbiBkZWZhdWx0IHBhcmFtcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGRlcGxveU5ld1N0YXRlTWFjaGluZShzdGFjayk7XG4gIGV4cGVjdChTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24oc3RhY2spKS50b01hdGNoU25hcHNob3QoKTtcbn0pO1xuXG50ZXN0KCdjaGVjayBkZXBsb3lDbG91ZFRyYWlsID0gZmFsc2UnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IHN0YXJ0U3RhdGUgPSBuZXcgc2ZuLlBhc3Moc3RhY2ssICdTdGFydFN0YXRlJyk7XG5cbiAgY29uc3QgcHJvcHM6IFMzVG9TdGVwRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBzdGF0ZU1hY2hpbmVQcm9wczoge1xuICAgICAgZGVmaW5pdGlvbjogc3RhcnRTdGF0ZVxuICAgIH0sXG4gICAgZGVwbG95Q2xvdWRUcmFpbDogZmFsc2VcbiAgfTtcblxuICBjb25zdCBjb25zdHJ1Y3QgPSBuZXcgUzNUb1N0ZXBGdW5jdGlvbihzdGFjaywgJ3Rlc3QtczMtc3RlcC1mdW5jdGlvbicsIHByb3BzKTtcblxuICBleHBlY3QoY29uc3RydWN0LmNsb3VkdHJhaWwgPT09IHVuZGVmaW5lZCk7XG59KTtcblxudGVzdCgnb3ZlcnJpZGUgZXZlbnRSdWxlUHJvcHMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IG15YnVja2V0ID0gbmV3IEJ1Y2tldChzdGFjaywgJ215YnVja2V0Jyk7XG4gIGNvbnN0IHN0YXJ0U3RhdGUgPSBuZXcgc2ZuLlBhc3Moc3RhY2ssICdTdGFydFN0YXRlJyk7XG5cbiAgY29uc3QgcHJvcHM6IFMzVG9TdGVwRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBzdGF0ZU1hY2hpbmVQcm9wczoge1xuICAgICAgZGVmaW5pdGlvbjogc3RhcnRTdGF0ZVxuICAgIH0sXG4gICAgZXhpc3RpbmdCdWNrZXRPYmo6IG15YnVja2V0LFxuICAgIGV2ZW50UnVsZVByb3BzOiB7XG4gICAgICBldmVudFBhdHRlcm46IHtcbiAgICAgICAgc291cmNlOiBbJ2F3cy5zMyddLFxuICAgICAgICBkZXRhaWxUeXBlOiBbJ0FXUyBBUEkgQ2FsbCB2aWEgQ2xvdWRUcmFpbCddLFxuICAgICAgICBkZXRhaWw6IHtcbiAgICAgICAgICBldmVudFNvdXJjZTogW1xuICAgICAgICAgICAgXCJzMy5hbWF6b25hd3MuY29tXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIGV2ZW50TmFtZTogW1xuICAgICAgICAgICAgXCJHZXRPYmplY3RcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgcmVxdWVzdFBhcmFtZXRlcnM6IHtcbiAgICAgICAgICAgIGJ1Y2tldE5hbWU6IFtcbiAgICAgICAgICAgICAgbXlidWNrZXQuYnVja2V0TmFtZVxuICAgICAgICAgICAgXVxuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgfTtcblxuICBuZXcgUzNUb1N0ZXBGdW5jdGlvbihzdGFjaywgJ3Rlc3QtczMtc3RlcC1mdW5jdGlvbicsIHByb3BzKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OkV2ZW50czo6UnVsZScsIHtcbiAgICBFdmVudFBhdHRlcm46IHtcbiAgICAgIFwic291cmNlXCI6IFtcbiAgICAgICAgXCJhd3MuczNcIlxuICAgICAgXSxcbiAgICAgIFwiZGV0YWlsLXR5cGVcIjogW1xuICAgICAgICBcIkFXUyBBUEkgQ2FsbCB2aWEgQ2xvdWRUcmFpbFwiXG4gICAgICBdLFxuICAgICAgXCJkZXRhaWxcIjoge1xuICAgICAgICBldmVudFNvdXJjZTogW1xuICAgICAgICAgIFwiczMuYW1hem9uYXdzLmNvbVwiXG4gICAgICAgIF0sXG4gICAgICAgIGV2ZW50TmFtZTogW1xuICAgICAgICAgIFwiR2V0T2JqZWN0XCJcbiAgICAgICAgXSxcbiAgICAgICAgcmVxdWVzdFBhcmFtZXRlcnM6IHtcbiAgICAgICAgICBidWNrZXROYW1lOiBbXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogXCJteWJ1Y2tldDE2MEY4MTMyXCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICB9LFxuICAgIFN0YXRlOiBcIkVOQUJMRURcIixcbiAgICBUYXJnZXRzOiBbXG4gICAgICB7XG4gICAgICAgIEFybjoge1xuICAgICAgICAgIFJlZjogXCJ0ZXN0czNzdGVwZnVuY3Rpb250ZXN0czNzdGVwZnVuY3Rpb25ldmVudHJ1bGVzdGVwZnVuY3Rpb25jb25zdHJ1Y3RTdGF0ZU1hY2hpbmVBNEQ0RjJCMFwiXG4gICAgICAgIH0sXG4gICAgICAgIElkOiBcIlRhcmdldDBcIixcbiAgICAgICAgUm9sZUFybjoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcInRlc3RzM3N0ZXBmdW5jdGlvbnRlc3RzM3N0ZXBmdW5jdGlvbmV2ZW50cnVsZXN0ZXBmdW5jdGlvbmNvbnN0cnVjdEV2ZW50c1J1bGVSb2xlMDdGRDdFQzFcIixcbiAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICBdXG4gIH0pO1xufSk7XG5cbnRlc3QoJ2NoZWNrIHByb3BlcnRpZXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IGNvbnN0cnVjdDogUzNUb1N0ZXBGdW5jdGlvbiA9IGRlcGxveU5ld1N0YXRlTWFjaGluZShzdGFjayk7XG5cbiAgZXhwZWN0KGNvbnN0cnVjdC5jbG91ZHRyYWlsICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5zdGF0ZU1hY2hpbmUgIT09IG51bGwpO1xuICBleHBlY3QoY29uc3RydWN0LnMzQnVja2V0ICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5jbG91ZHdhdGNoQWxhcm1zICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5zdGF0ZU1hY2hpbmVMb2dHcm91cCAhPT0gbnVsbCk7XG4gIGV4cGVjdChjb25zdHJ1Y3QuczNMb2dnaW5nQnVja2V0ICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5jbG91ZHRyYWlsICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5jbG91ZHRyYWlsQnVja2V0ICE9PSBudWxsKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5jbG91ZHRyYWlsTG9nZ2luZ0J1Y2tldCAhPT0gbnVsbCk7XG59KTtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbi8vIFRlc3QgYmFkIGNhbGwgd2l0aCBleGlzdGluZ0J1Y2tldCBhbmQgYnVja2V0UHJvcHNcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KFwiVGVzdCBiYWQgY2FsbCB3aXRoIGV4aXN0aW5nQnVja2V0IGFuZCBidWNrZXRQcm9wc1wiLCAoKSA9PiB7XG4gIC8vIFN0YWNrXG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gIGNvbnN0IHRlc3RCdWNrZXQgPSBuZXcgQnVja2V0KHN0YWNrLCAndGVzdC1idWNrZXQnLCB7fSk7XG4gIGNvbnN0IHN0YXJ0U3RhdGUgPSBuZXcgc2ZuLlBhc3Moc3RhY2ssICdTdGFydFN0YXRlJyk7XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIC8vIEhlbHBlciBkZWNsYXJhdGlvblxuICAgIG5ldyBTM1RvU3RlcEZ1bmN0aW9uKHN0YWNrLCBcImJhZC1zMy1hcmdzXCIsIHtcbiAgICAgIHN0YXRlTWFjaGluZVByb3BzOiB7XG4gICAgICAgIGRlZmluaXRpb246IHN0YXJ0U3RhdGVcbiAgICAgIH0sXG4gICAgICBleGlzdGluZ0J1Y2tldE9iajogdGVzdEJ1Y2tldCxcbiAgICAgIGJ1Y2tldFByb3BzOiB7XG4gICAgICAgIHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5LkRFU1RST1lcbiAgICAgIH0sXG4gICAgfSk7XG4gIH07XG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoKTtcbn0pOyJdfQ==