#  Copyright (c) 2019 JD Williams
#
#  This file is part of Firefly, a Python SOA framework built by JD Williams. Firefly is free software; you can
#  redistribute it and/or modify it under the terms of the GNU General Public License as published by the
#  Free Software Foundation; either version 3 of the License, or (at your option) any later version.
#
#  Firefly is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
#  Public License for more details. You should have received a copy of the GNU Lesser General Public
#  License along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
#  You should have received a copy of the GNU General Public License along with Firefly. If not, see
#  <http://www.gnu.org/licenses/>.

"""Applies a fix to CR LF TAB handling in xml.dom.

Fixes this: http://code.google.com/p/chromium/issues/detail?id=76293
Working around this: http://bugs.python.org/issue5752
TODO(bradnelson): Consider dropping this when we drop XP support.
"""


import xml.dom.minidom


def _Replacement_write_data(writer, data, is_attrib=False):
  """Writes datachars to writer."""
  data = data.replace("&", "&amp;").replace("<", "&lt;")
  data = data.replace("\"", "&quot;").replace(">", "&gt;")
  if is_attrib:
    data = data.replace(
        "\r", "&#xD;").replace(
        "\n", "&#xA;").replace(
        "\t", "&#x9;")
  writer.write(data)


def _Replacement_writexml(self, writer, indent="", addindent="", newl=""):
  # indent = current indentation
  # addindent = indentation to add to higher levels
  # newl = newline string
  writer.write(indent+"<" + self.tagName)

  attrs = self._get_attributes()
  a_names = attrs.keys()
  a_names.sort()

  for a_name in a_names:
    writer.write(" %s=\"" % a_name)
    _Replacement_write_data(writer, attrs[a_name].value, is_attrib=True)
    writer.write("\"")
  if self.childNodes:
    writer.write(">%s" % newl)
    for node in self.childNodes:
      node.writexml(writer, indent + addindent, addindent, newl)
    writer.write("%s</%s>%s" % (indent, self.tagName, newl))
  else:
    writer.write("/>%s" % newl)


class XmlFix(object):
  """Object to manage temporary patching of xml.dom.minidom."""

  def __init__(self):
    # Preserve current xml.dom.minidom functions.
    self.write_data = xml.dom.minidom._write_data
    self.writexml = xml.dom.minidom.Element.writexml
    # Inject replacement versions of a function and a method.
    xml.dom.minidom._write_data = _Replacement_write_data
    xml.dom.minidom.Element.writexml = _Replacement_writexml

  def Cleanup(self):
    if self.write_data:
      xml.dom.minidom._write_data = self.write_data
      xml.dom.minidom.Element.writexml = self.writexml
      self.write_data = None

  def __del__(self):
    self.Cleanup()
