from __future__ import annotations

import contextlib
import os
import pathlib
import random
import subprocess
import sys
import sysconfig
import time
from typing import Any, AsyncIterable, Awaitable, Callable, Dict, Iterator, List

import pytest
import pytest_asyncio

# https://github.com/pytest-dev/pytest/issues/7469
from _pytest.fixtures import SubRequest

from chinilla.data_layer.data_layer_util import NodeType, Status
from chinilla.data_layer.data_store import DataStore
from chinilla.types.blockchain_format.tree_hash import bytes32
from tests.core.data_layer.util import (
    ChinillaRoot,
    Example,
    add_0123_example,
    add_01234567_example,
    create_valid_node_values,
)

# TODO: These are more general than the data layer and should either move elsewhere or
#       be replaced with an existing common approach.  For now they can at least be
#       shared among the data layer test files.


@pytest.fixture(name="scripts_path", scope="session")
def scripts_path_fixture() -> pathlib.Path:
    scripts_string = sysconfig.get_path("scripts")
    if scripts_string is None:
        raise Exception("These tests depend on the scripts path existing")

    return pathlib.Path(scripts_string)


@pytest.fixture(name="chinilla_root", scope="function")
def chinilla_root_fixture(tmp_path: pathlib.Path, scripts_path: pathlib.Path) -> ChinillaRoot:
    root = ChinillaRoot(path=tmp_path.joinpath("chinilla_root"), scripts_path=scripts_path)
    root.run(args=["init"])
    root.run(args=["configure", "--set-log-level", "INFO"])

    return root


@contextlib.contextmanager
def closing_chinilla_root_popen(chinilla_root: ChinillaRoot, args: List[str]) -> Iterator[None]:
    environment = {**os.environ, "CHINILLA_ROOT": os.fspath(chinilla_root.path)}

    with subprocess.Popen(args=args, env=environment) as process:
        try:
            yield
        finally:
            process.terminate()
            try:
                process.wait(timeout=10)
            except subprocess.TimeoutExpired:
                process.kill()


@pytest.fixture(name="chinilla_daemon", scope="function")
def chinilla_daemon_fixture(chinilla_root: ChinillaRoot) -> Iterator[None]:
    with closing_chinilla_root_popen(
        chinilla_root=chinilla_root, args=[sys.executable, "-m", "chinilla.daemon.server"]
    ):
        # TODO: this is not pretty as a hard coded time
        # let it settle
        time.sleep(5)
        yield


@pytest.fixture(name="chinilla_data", scope="function")
def chinilla_data_fixture(
    chinilla_root: ChinillaRoot, chinilla_daemon: None, scripts_path: pathlib.Path
) -> Iterator[None]:
    with closing_chinilla_root_popen(
        chinilla_root=chinilla_root, args=[os.fspath(scripts_path.joinpath("chinilla_data_layer"))]
    ):
        # TODO: this is not pretty as a hard coded time
        # let it settle
        time.sleep(5)
        yield


@pytest.fixture(name="create_example", params=[add_0123_example, add_01234567_example])
def create_example_fixture(request: SubRequest) -> Callable[[DataStore, bytes32], Awaitable[Example]]:
    # https://github.com/pytest-dev/pytest/issues/8763
    return request.param  # type: ignore[no-any-return]


@pytest.fixture(name="database_uri")
def database_uri_fixture() -> str:
    return f"file:db_{random.randint(0, 99999999)}?mode=memory&cache=shared"


@pytest.fixture(name="tree_id", scope="function")
def tree_id_fixture() -> bytes32:
    base = b"a tree id"
    pad = b"." * (32 - len(base))
    return bytes32(pad + base)


@pytest_asyncio.fixture(name="raw_data_store", scope="function")
async def raw_data_store_fixture(database_uri: str) -> AsyncIterable[DataStore]:
    store = await DataStore.create(database=database_uri, uri=True)
    yield store
    await store.close()


@pytest_asyncio.fixture(name="data_store", scope="function")
async def data_store_fixture(raw_data_store: DataStore, tree_id: bytes32) -> AsyncIterable[DataStore]:
    await raw_data_store.create_tree(tree_id=tree_id, status=Status.COMMITTED)

    await raw_data_store.check()
    yield raw_data_store
    await raw_data_store.check()


@pytest.fixture(name="node_type", params=NodeType)
def node_type_fixture(request: SubRequest) -> NodeType:
    return request.param  # type: ignore[no-any-return]


@pytest_asyncio.fixture(name="valid_node_values")
async def valid_node_values_fixture(
    data_store: DataStore,
    tree_id: bytes32,
    node_type: NodeType,
) -> Dict[str, Any]:
    await add_01234567_example(data_store=data_store, tree_id=tree_id)
    node_a = await data_store.get_node_by_key(key=b"\x02", tree_id=tree_id)
    node_b = await data_store.get_node_by_key(key=b"\x04", tree_id=tree_id)

    return create_valid_node_values(node_type=node_type, left_hash=node_a.hash, right_hash=node_b.hash)


@pytest.fixture(name="bad_node_type", params=range(2 * len(NodeType)))
def bad_node_type_fixture(request: SubRequest, valid_node_values: Dict[str, Any]) -> int:
    if request.param == valid_node_values["node_type"]:
        pytest.skip("Actually, this is a valid node type")

    return request.param  # type: ignore[no-any-return]
