import re
import redis
import time
import json as Json
from random import shuffle
from hashlib import md5
from functools import wraps
from collections import defaultdict
from collections.abc import Iterable, Callable


def fn_timer(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        start = time.time()
        result = func(*args, **kwargs)
        end = time.time()
        print("Function [{}] Spend {:.3f} s".format(func.__name__, end - start))
        return result

    return wrapper


def _flatten(item):
    for k, v in item.items():
        if isinstance(v, dict):
            yield from _flatten(v)
        yield k, v


def json_to_dict(json):
    if isinstance(json, dict): return json

    if not json: return {}
    json = json.replace('\'', '"')
    try:
        return Json.loads(json)
    except:
        print(f'Invalid json data: {json}')
        return {}


def body_to_dict(data):
    if isinstance(data, dict): return data

    if not data: return {}
    assert '=' in data, f'Invalid data: {data}'
    return dict(_.split('=', 1) for _ in data.split('&'))


def _spilt_url(url):
    if not url: return {}
    path = url.split('?', 1)
    return [path[0], ''] if len(path) == 1 else path


def url_to_dict(url):
    if not url: return {}

    url = url.replace('://', '/')

    _path, _param = _spilt_url(url)
    protocol, domain, *path = _path.split('/')
    if _param:
        if '=' in _param:
            param = dict(p.split('=', 1) for p in _param.split('&'))
        else:
            param = {_param: _param}
    else:
        param = {}

    return {
        'protocol': protocol,
        'domain': domain,
        'path': path,
        'param': param
    }


def headers_to_dict(headers):
    if isinstance(headers, dict): return headers

    if not headers: return {}
    return {_.split(':', 1)[0].strip(): _.split(':', 1)[1].strip() for _ in headers.split('\n') if
            len(_.split(':', 1)) == 2}


def cookies_to_dict(cookies):
    if isinstance(cookies, dict): return cookies

    if not cookies: return {}
    return {_.split('=', 1)[0].strip(): _.split('=', 1)[1].strip() for _ in cookies.strip('\n;').split(';')}


def dict_to_body(data: dict):
    return '&'.join([f'{key}={value}' for key, value in data.items()])


def dict_to_json(json: dict):
    return Json.dumps(json)


def search(key, data=None, target_type=None):
    my_dict = defaultdict(list)
    for k, v in _flatten(data):
        my_dict[k].append(v)

    if isinstance(key, Iterable) and not isinstance(key, (str, bytes)):
        return {
            k: [_ for _ in my_dict.get(k) if isinstance(_, target_type)] if target_type else my_dict.get(k)
            for k in key
        }
    else:
        result = [_ for _ in my_dict.get(key) if isinstance(_, target_type)] if target_type else my_dict.get(key)
        return result[0] if result and len(result) == 1 else result


def strip(*args, data=None, strip_key=False):
    if not data: args, data = args[:-1], args[-1]

    for st_key in args:
        if isinstance(st_key, (str, Callable)): st_key = [st_key]

        for r in st_key:
            result = {}
            for key, value in data.items():
                key, value = _strip(key, value, r, strip_key=strip_key)
                result[key] = value
            data = result

    return data


def replace(replace_map=None, data=None, replace_key=False):
    assert isinstance(data, dict), 'item must be dict'

    for r_key, r_value in replace_map.items():
        result = {}
        for key, value in data.items():
            key = key if not replace_key else key.replace(r_key, r_value)
            if isinstance(value, str):
                result[key] = value.replace(r_key, r_value)
            elif isinstance(value, dict):
                result[key] = replace(data=value, replace_key=replace_key, replace_map={r_key: r_value})
            elif isinstance(value, list):
                result[key] = _process_list(key, value, rule=(r_key, r_value), process_key=replace_key)
            else:
                result[key] = value
        data = result

    return data


def update(update_map, data=None, target_type=None):
    assert isinstance(data, dict), 'item must be dict'
    if not target_type: target_type = (str, bytes, int, float, list, dict)

    for u_key, u_value in update_map.items():
        result = {}
        for key, value in data.items():
            if isinstance(value, target_type) and not isinstance(value, dict):
                result[key] = u_value if key == u_key else value
            elif isinstance(value, dict):
                result[key] = update(update_map={u_key: u_value}, data=value, target_type=target_type)
            else:
                result[key] = value
        data = result

    return data


def delete(*args, data=None, target_type=None):
    if not data: args, data = args[:-1], args[-1]
    if not target_type: target_type = (str, bytes, int, float, list, dict)

    for d_key in args:
        assert isinstance(d_key, (str, list, tuple)), f'args must be str、list or tuple, get {d_key}'

        if isinstance(d_key, str): d_key = [d_key]
        for d_k in d_key:
            result = {}
            for key, value in data.items():
                if isinstance(value, target_type):
                    if key == d_k:
                        continue
                    else:
                        result[key] = value
                elif isinstance(value, dict):
                    result[key] = delete(d_k, data=value, target_type=target_type)
                else:
                    result[key] = value
            data = result

    return data


def _strip(key, value, rule, strip_key=False):
    if type(rule).__name__ == 'function':
        rule, switch = rule(key, value)
        if not switch: return key, value

    key = key.replace(rule, '') if strip_key else key

    if isinstance(value, (str, int, float)):
        value = value if not isinstance(value, str) else value.replace(rule, '')
    elif isinstance(value, dict):
        value = strip(rule, data=value, strip_key=strip_key)
    elif isinstance(value, list):
        value = _process_list(key, value, rule, process_key=strip_key)

    return key, value


def _process_list(key, value, rule, process_key=False):
    s = False
    if isinstance(rule, str): rule, s = (rule, ''), True

    result = []
    for v in value:
        if isinstance(v, str):
            result.append(v.replace(*rule))
        elif isinstance(v, list):
            if s:
                v = _strip(key, v, rule[0], strip_key=process_key)
            else:
                v = replace(replace_key=process_key, data={'_': v}, replace_map={rule[0]: rule[1]})

            result.append(v)
        elif isinstance(v, dict):
            if s:
                v = strip(rule, data=v, strip_key=process_key)
            else:
                v = replace(replace_key=process_key, data=v, replace_map={rule[0]: rule[1]})
            result.append(v)
        else:
            result.append(v)

    return result


def flatten(data):
    for k, v in data.items():
        if isinstance(v, dict):
            yield from flatten(v)
        else:
            yield k, v


def re_search(re_map, data, flags=None, index=None):
    s = False
    if isinstance(re_map, str): re_map, s = {'_': re_map}, True
    result = {}
    for key, pattern in re_map.items():
        if isinstance(pattern, str):
            r = re.search(pattern, data, flags=flags or 0)
            if not r and not flags:
                r = re.search(pattern, data, flags=re.S)
        elif isinstance(pattern, re.Pattern):
            r = pattern.search(data)
        elif isinstance(pattern, dict):
            r = re_search(pattern, data, flags=flags, index=index)
        else:
            raise Exception(f'Type Error ... re_search not support {type(pattern)}')

        result[key] = r

    result_g = _get_group_data(result, index=index)

    return result_g if not s else result_g.get('_')


def _get_group_data(data, index=None):
    result_g = {}
    for k, v in data.items():
        if v and isinstance(v, dict):
            result_g[k] = _get_group_data(v)
        elif isinstance(v, str):
            result_g[k] = v
        else:
            try:
                result_g[k] = v.group(index or 0) if v else ''
            except IndexError:
                result_g[k] = v.group()

    return result_g


def re_findall(re_map, data, flags=None, iter=False):
    s = False
    if isinstance(re_map, str): re_map, s = {'_': re_map}, True

    result = {}
    for key, pattern in re_map.items():
        if isinstance(pattern, str):
            r = re.finditer(pattern, data, flags=flags or 0) if iter else re.findall(pattern, data, flags=flags or 0)
            if not r and not flags:
                r = re.search(pattern, data, flags=re.S)
        elif isinstance(pattern, re.Pattern):
            r = pattern.finditer(data) if iter else pattern.findall(data)
        elif isinstance(pattern, dict):
            r = re_findall(pattern, data, flags=flags, iter=iter)
        else:
            raise Exception(f'Type Error ... re_search not support {type(pattern)}')

        result[key] = r

    return result if not s else result.get('_')


def merge(*args, overwrite=False):
    default_dict = defaultdict(list)

    v_dict = defaultdict(list)
    for d in args:
        if not isinstance(d, dict): continue
        for k, v in d.items():
            if isinstance(v, dict):
                v_dict[k].append(v)
                continue
            if overwrite and default_dict.get(k) and v in default_dict.get(k): continue
            default_dict[k].append(v)

    for k, v in v_dict.items():
        default_dict[k].append(merge(*v, overwrite=overwrite))

    return {k: v[0] if k in v_dict.keys() else v for k, v in dict(default_dict).items()}


def args_split(args: tuple):
    arg = tuple(i for i in args if not isinstance(i, dict))
    kwarg = [i for i in args if isinstance(i, dict)]

    if len(kwarg) >= 1:
        _kwargs = {}
        for i in kwarg:
            _kwargs.update(i)
        kwarg = _kwargs

    return arg, kwarg or {}


def random_list(stop, start=0, step=1):
    numbers = list(range(start, stop + start, step))
    shuffle(numbers)
    return numbers


def human_time(tm):
    days, hours, minutes, seconds = 0, 0, 0, 0

    if tm * 1000 < 1:
        ms = tm * 1000
    else:
        s, ms = int(str(tm).split('.')[0]), float('0.' + str(tm).split('.')[1]) * 1000
        if tm < 60:
            seconds = (s % 60)
        elif tm < 60 * 60:
            minutes = (s % 3600) // 60
            seconds = (s % 60)
        elif tm < 60 * 60 * 24:
            hours = (s % 86400) // 3600
            minutes = (s % 3600) // 60
            seconds = (s % 60)
        else:
            days = s // 86400
            hours = (s % 86400) // 3600
            minutes = (s % 3600) // 60
            seconds = (s % 60)

    try:
        s = '{} {}:{}:{},{:.2e}'.format(days, hours, minutes, seconds, ms)
    except:
        print()
    else:
        return s


def get_md5(*args):
    """
    @summary: 获取唯一的32位md5
    ---------
    @param *args: 参与联合去重的值
    ---------
    @result: 7c8684bcbdfcea6697650aa53d7b1405
    """

    m = md5()
    for arg in args:
        m.update(str(arg).encode())

    return m.hexdigest()


# 根据 开辟内存大小 和 种子，生成不同的hash函数
# 也就是构造上述提到的：Bloom Filter使用k个相互独立的哈希函数，我们记为 **H = { H1( ),  H2( ),  ...,  Hk( ) }**
class SimpleHash(object):
    def __init__(self, bitSize, seed):
        self.bitSize = bitSize
        self.seed = seed

    def hash(self, value):
        ret = 0
        for i in range(len(value)):
            # print(f"value[i] = {value[i]},  ord(value[i]) = {ord(value[i])}")
            ret += self.seed * ret + ord(value[i])
        # 控制hashValue的值在这个内存空间的范围
        hashValue = (self.bitSize - 1) & ret
        # print(f"value = {value}, hashValue = {hashValue}")
        return hashValue


# 在redis中初始化一个大字符串，也可以认为是在redis中开辟了一块内存空间
# 需要指定数据库名， 比如这儿用到的就是db2
# 指定使用数据块个数，也就是开辟几个这样的大字符串。
# 当数据达到非常大时，512M肯定是不够用的，可能每个位都被置为1了，所以需要开辟多个大字符串
# 大字符串名name = (key + int)
class BloomFilter(object):
    def __init__(self, host='localhost', port=6379, db=2, blockNum=1, key='bloomfilter'):
        """
        :param host: the host of Redis
        :param port: the port of Redis
        :param db: witch db in Redis
        :param blockNum: one blockNum for about 90,000,000; if you have more strings for filtering, increase it.
        :param key: the key's name in Redis
        """
        self.server = redis.Redis(host=host, port=port, db=db)
        # 2^31 = 256M
        # 这是一个限制值，最大为256M，因为在redis中，字符串值可以进行伸展，伸展时，空白位置以0填充。
        self.bit_size = 1 << 31  # Redis的String类型最大容量为512M，现使用256M
        self.seeds = [5, 7, 11, 13, 31, 37, 61]
        self.key = key
        self.blockNum = blockNum
        self.hashfunc = []
        for seed in self.seeds:
            # 根据seed 构造出 k=7 个独立的hash函数
            self.hashfunc.append(SimpleHash(self.bit_size, seed))

    # 判断元素是否在集合中
    def isContains(self, str_input):
        if not str_input:
            return False
        m5 = md5()
        m5.update(str_input.encode('utf-8'))
        # 先取目标字符串的md5值
        str_input = m5.hexdigest()
        ret = True
        name = self.key + str(int(str_input[0:2], 16) % self.blockNum)
        for f in self.hashfunc:
            loc = f.hash(str_input)
            ret = ret & self.server.getbit(name, loc)
        return ret

    # 将str_input映射的结果，写入到大字符串中，也就是置上相关的标志位
    def insert(self, str_input):
        m5 = md5()
        m5.update(str_input.encode('utf-8'))
        str_input = m5.hexdigest()
        name = self.key + str(int(str_input[0:2], 16) % self.blockNum)
        for f in self.hashfunc:
            loc = f.hash(str_input)
            # print(f"name = {name}, loc = {loc}")
            self.server.setbit(name, loc, 1)


def match_list(string, slist):
    for s in slist:
        if len(s) > len(string) and s.endswith(string):
            return True
        elif string.endswith(s):
            return True
        else:
            continue
    return False


def get_traceback_info(tb):
    func_name = tb.tb_frame.f_code.co_name
    lineno = tb.f_lineno if hasattr(tb, 'f_lineno') else tb.tb_frame.f_lineno
    filepath = tb.tb_frame.f_code.co_filename
    return func_name, lineno, filepath
