/***************************************************************************************************
 * Copyright (c) 2017 - 2022 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice, this
 * list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 **************************************************************************************************/
/*! \file
  \brief Basic subset of epilogue functionality for supporting StreamK decompositions
*/


#pragma once

#include "cutlass/cutlass.h"
#include "cutlass/functional.h"
#include "cutlass/block_striped.h"

////////////////////////////////////////////////////////////////////////////////

namespace cutlass {
namespace epilogue {
namespace threadblock {

////////////////////////////////////////////////////////////////////////////////


/// StreamK epilogue functionality for cross-block accumulator fragment reduction
template <
  typename Shape,                          ///< Shape of threadblock tile (concept: GemmShape)
  int PartitionsK,
  typename WarpMmaOperator,                ///< Warp-level MMA operator (concept: gemm::warp::MmaTensorOp)
  typename AccumulatorFragmentIterator>    ///< Fragment iterator selecting accumulators
class EpilogueBaseStreamK
{

protected:

  /// The complete warp-level accumulator tile
  using AccumulatorTile = typename AccumulatorFragmentIterator::AccumulatorTile;

  /// Number of warps
  using WarpCount = gemm::GemmShape<
                        Shape::kM / WarpMmaOperator::Shape::kM,
                        Shape::kN / WarpMmaOperator::Shape::kN, PartitionsK>;

  /// Number of threads per block
  static int const kBlockThreads = 32 * WarpCount::kCount;

  /// Numerical accumulation element type
  using ElementAccumulator = typename WarpMmaOperator::ElementC;

  /// Fragment type used by the accumulator tile's fragment iterator
  using AccumulatorFragment = typename AccumulatorFragmentIterator::Fragment;

  /// Block-striped transfer utility for sharing AccumulatorFragment
  using BlockStripedT = BlockStriped<kBlockThreads, AccumulatorFragment, ElementAccumulator>;

  /// Number of elements per fragment
  static int const kFragmentElements = sizeof(AccumulatorFragment) / sizeof(ElementAccumulator);

public:

  /// Number of fragments per accumulator tile
  static int const kAccumulatorFragments = AccumulatorFragmentIterator::Policy::kIterations;

  /// Number of workspace accumulation elements shared per output tile
  static int const kPeerAccumulators = WarpMmaOperator::Shape::kMN * WarpCount::kCount;

protected:

  /// ElementAccumulator stride in the shared workspace between different peer blocks (two: each peer block can share accumulators for up to two tiles)
  static const int kPeerStride = kPeerAccumulators * 2;


public:

  /// Thread index in the threadblock
  int thread_idx;

public:

  /// Constructor
  CUTLASS_DEVICE
  EpilogueBaseStreamK(
      int thread_idx)                                       ///< ID of a thread within the threadblock
  :
      thread_idx(thread_idx)
  {}


  /// Aggregates the accumulator sets shared by peer blocks in the global workspace
  CUTLASS_DEVICE
  void reduce(
      AccumulatorFragment &accum_fragment,                  ///< [out] sum of all shared accumulator fragments for these peer partials
      int peer_idx_begin,
      int peer_idx_end,
      int reduce_fragment_idx,
      ElementAccumulator *element_workspace)
  {
    plus<AccumulatorFragment> add_fragments;

    int accum_set_offset =
        (peer_idx_begin * kPeerStride) +
        (reduce_fragment_idx * kBlockThreads * kFragmentElements);

    // Load first peer fragment
    BlockStripedT::load(accum_fragment, element_workspace + accum_set_offset, this->thread_idx);

    accum_set_offset += kPeerStride;            // Move to next peer
    accum_set_offset += kPeerAccumulators;      // Move to non-starting accumulator set for peer

    // Reduce additional peer fragments
    #pragma unroll 2
    while (accum_set_offset < peer_idx_end * kPeerStride)
    {
      AccumulatorFragment addend_fragment;
      BlockStripedT::load(addend_fragment, element_workspace + accum_set_offset, this->thread_idx);
      accum_set_offset += kPeerStride;

      accum_fragment = add_fragments(accum_fragment, addend_fragment);
    }
  }


  /// Shares the accumulator set with peers in the global workspace
  CUTLASS_DEVICE
  void share(
      int peer_idx,
      ElementAccumulator *element_workspace,          ///< Output pointer for writing this block's accumulator set to
      AccumulatorTile const &accumulators,            ///< Complete warp-level accumulator tile
      bool started_tile)
  {
    int accum_set_offset = peer_idx * kPeerStride;

    if (!started_tile) {
      // Move to non-starting accumulator set
      accum_set_offset += kPeerAccumulators;
    }

    AccumulatorFragmentIterator accum_fragment_iterator(accumulators);

    CUTLASS_PRAGMA_UNROLL
    for (int iter = 0; iter < kAccumulatorFragments; ++iter)
    {
      // Acquire reordered accumulator fragment
      AccumulatorFragment accum_fragment;
      accum_fragment_iterator.load(accum_fragment);
      ++accum_fragment_iterator;

      // Store accumulator fragment
      BlockStripedT::store(element_workspace + accum_set_offset, accum_fragment, this->thread_idx);

      accum_set_offset += (kFragmentElements * kBlockThreads);
    }
  }

};



////////////////////////////////////////////////////////////////////////////////

} // namespace threadblock
} // namespace epilogue
} // namespace cutlass

////////////////////////////////////////////////////////////////////////////////
