"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpsRedirect = void 0;
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const crypto = require("crypto");
/**
 * Allows creating a domainA -> domainB redirect using CloudFront and S3.
 * You can specify multiple domains to be redirected.
 */
class HttpsRedirect extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const domainNames = props.recordNames || [props.zone.zoneName];
        const redirectCertArn = props.certificate ? props.certificate.certificateArn : new aws_certificatemanager_1.DnsValidatedCertificate(this, 'RedirectCertificate', {
            domainName: domainNames[0],
            subjectAlternativeNames: domainNames,
            hostedZone: props.zone,
            region: 'us-east-1',
        }).certificateArn;
        const redirectBucket = new aws_s3_1.Bucket(this, 'RedirectBucket', {
            websiteRedirect: {
                hostName: props.targetDomain,
                protocol: aws_s3_1.RedirectProtocol.HTTPS,
            },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        });
        const redirectDist = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'RedirectDistribution', {
            defaultRootObject: '',
            originConfigs: [{
                    behaviors: [{ isDefaultBehavior: true }],
                    customOriginSource: {
                        domainName: redirectBucket.bucketWebsiteDomainName,
                        originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY,
                    },
                }],
            aliasConfiguration: {
                acmCertRef: redirectCertArn,
                names: domainNames,
            },
            comment: `Redirect to ${props.targetDomain} from ${domainNames.join(', ')}`,
            priceClass: aws_cloudfront_1.PriceClass.PRICE_CLASS_ALL,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        });
        domainNames.forEach((domainName) => {
            const hash = crypto.createHash('md5').update(domainName).digest('hex').substr(0, 6);
            new aws_route53_1.ARecord(this, `RedirectAliasRecord${hash}`, {
                recordName: domainName,
                zone: props.zone,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(redirectDist)),
            });
        });
    }
}
exports.HttpsRedirect = HttpsRedirect;
//# sourceMappingURL=data:application/json;base64,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