"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginRequestQueryStringBehavior = exports.OriginRequestHeaderBehavior = exports.OriginRequestCookieBehavior = exports.OriginRequestPolicy = void 0;
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * (experimental) A Origin Request Policy configuration.
 *
 * @experimental
 * @resource AWS::CloudFront::OriginRequestPolicy
 */
class OriginRequestPolicy extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d;
        super(scope, id, {
            physicalName: props.originRequestPolicyName,
        });
        const originRequestPolicyName = (_a = props.originRequestPolicyName) !== null && _a !== void 0 ? _a : core_1.Names.uniqueId(this);
        if (!core_1.Token.isUnresolved(originRequestPolicyName) && !originRequestPolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'originRequestPolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.originRequestPolicyName}'`);
        }
        const cookies = (_b = props.cookieBehavior) !== null && _b !== void 0 ? _b : OriginRequestCookieBehavior.none();
        const headers = (_c = props.headerBehavior) !== null && _c !== void 0 ? _c : OriginRequestHeaderBehavior.none();
        const queryStrings = (_d = props.queryStringBehavior) !== null && _d !== void 0 ? _d : OriginRequestQueryStringBehavior.none();
        const resource = new cloudfront_generated_1.CfnOriginRequestPolicy(this, 'Resource', {
            originRequestPolicyConfig: {
                name: originRequestPolicyName,
                comment: props.comment,
                cookiesConfig: {
                    cookieBehavior: cookies.behavior,
                    cookies: cookies.cookies,
                },
                headersConfig: {
                    headerBehavior: headers.behavior,
                    headers: headers.headers,
                },
                queryStringsConfig: {
                    queryStringBehavior: queryStrings.behavior,
                    queryStrings: queryStrings.queryStrings,
                },
            },
        });
        this.originRequestPolicyId = resource.ref;
    }
    /**
     * (experimental) Imports a Origin Request Policy from its id.
     *
     * @experimental
     */
    static fromOriginRequestPolicyId(scope, id, originRequestPolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.originRequestPolicyId = originRequestPolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed origin request policy. */
    static fromManagedOriginRequestPolicy(managedOriginRequestPolicyId) {
        return new class {
            constructor() {
                this.originRequestPolicyId = managedOriginRequestPolicyId;
            }
        }();
    }
}
exports.OriginRequestPolicy = OriginRequestPolicy;
/**
 * (experimental) This policy includes only the User-Agent and Referer headers.
 *
 * It doesn’t include any query strings or cookies.
 *
 * @experimental
 */
OriginRequestPolicy.USER_AGENT_REFERER_HEADERS = OriginRequestPolicy.fromManagedOriginRequestPolicy('acba4595-bd28-49b8-b9fe-13317c0390fa');
/**
 * (experimental) This policy includes the header that enables cross-origin resource sharing (CORS) requests when the origin is a custom origin.
 *
 * @experimental
 */
OriginRequestPolicy.CORS_CUSTOM_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('59781a5b-3903-41f3-afcb-af62929ccde1');
/**
 * (experimental) This policy includes the headers that enable cross-origin resource sharing (CORS) requests when the origin is an Amazon S3 bucket.
 *
 * @experimental
 */
OriginRequestPolicy.CORS_S3_ORIGIN = OriginRequestPolicy.fromManagedOriginRequestPolicy('88a5eaf4-2fd4-4709-b370-b4c650ea3fcf');
/**
 * (experimental) This policy includes all values (query strings, headers, and cookies) in the viewer request.
 *
 * @experimental
 */
OriginRequestPolicy.ALL_VIEWER = OriginRequestPolicy.fromManagedOriginRequestPolicy('216adef6-5c7f-47e4-b989-5492eafa07d3');
/**
 * (experimental) This policy is designed for use with an origin that is an AWS Elemental MediaTailor endpoint.
 *
 * @experimental
 */
OriginRequestPolicy.ELEMENTAL_MEDIA_TAILOR = OriginRequestPolicy.fromManagedOriginRequestPolicy('775133bc-15f2-49f9-abea-afb2e0bf67d2');
/**
 * (experimental) Ddetermines whether any cookies in viewer requests (and if so, which cookies) are included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class OriginRequestCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * (experimental) Cookies in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any cookies that are listed in a CachePolicy are still included in origin requests.
     *
     * @experimental
     */
    static none() { return new OriginRequestCookieBehavior('none'); }
    /**
     * (experimental) All cookies in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static all() { return new OriginRequestCookieBehavior('all'); }
    /**
     * (experimental) Only the provided `cookies` are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new OriginRequestCookieBehavior('whitelist', cookies);
    }
}
exports.OriginRequestCookieBehavior = OriginRequestCookieBehavior;
/**
 * (experimental) Determines whether any HTTP headers (and if so, which headers) are included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class OriginRequestHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * (experimental) HTTP headers are not included in requests that CloudFront sends to the origin.
     *
     * Any headers that are listed in a CachePolicy are still included in origin requests.
     *
     * @experimental
     */
    static none() { return new OriginRequestHeaderBehavior('none'); }
    /**
     * (experimental) All HTTP headers in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * Additionally, any additional CloudFront headers provided are included; the additional headers are added by CloudFront.
     *
     * @see https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-cloudfront-headers.html
     * @experimental
     */
    static all(...cloudfrontHeaders) {
        if (cloudfrontHeaders.length > 0) {
            if (!cloudfrontHeaders.every(header => header.startsWith('CloudFront-'))) {
                throw new Error('additional CloudFront headers passed to `OriginRequestHeaderBehavior.all()` must begin with \'CloudFront-\'');
            }
            return new OriginRequestHeaderBehavior('allViewerAndWhitelistCloudFront', cloudfrontHeaders);
        }
        else {
            return new OriginRequestHeaderBehavior('allViewer');
        }
    }
    /**
     * (experimental) Listed headers are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        return new OriginRequestHeaderBehavior('whitelist', headers);
    }
}
exports.OriginRequestHeaderBehavior = OriginRequestHeaderBehavior;
/**
 * (experimental) Determines whether any URL query strings in viewer requests (and if so, which query strings) are included in requests that CloudFront sends to the origin.
 *
 * @experimental
 */
class OriginRequestQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * (experimental) Query strings in viewer requests are not included in requests that CloudFront sends to the origin.
     *
     * Any query strings that are listed in a CachePolicy are still included in origin requests.
     *
     * @experimental
     */
    static none() { return new OriginRequestQueryStringBehavior('none'); }
    /**
     * (experimental) All query strings in viewer requests are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static all() { return new OriginRequestQueryStringBehavior('all'); }
    /**
     * (experimental) Only the provided `queryStrings` are included in requests that CloudFront sends to the origin.
     *
     * @experimental
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new OriginRequestQueryStringBehavior('whitelist', queryStrings);
    }
}
exports.OriginRequestQueryStringBehavior = OriginRequestQueryStringBehavior;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoib3JpZ2luLXJlcXVlc3QtcG9saWN5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsib3JpZ2luLXJlcXVlc3QtcG9saWN5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUF1RDtBQUV2RCxpRUFBZ0U7Ozs7Ozs7QUF5RGhFLE1BQWEsbUJBQW9CLFNBQVEsZUFBUTs7OztJQTZCL0MsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFrQyxFQUFFOztRQUM1RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFlBQVksRUFBRSxLQUFLLENBQUMsdUJBQXVCO1NBQzVDLENBQUMsQ0FBQztRQUVILE1BQU0sdUJBQXVCLFNBQUcsS0FBSyxDQUFDLHVCQUF1QixtQ0FBSSxZQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RGLElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDLHVCQUF1QixDQUFDLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLEVBQUU7WUFDL0YsTUFBTSxJQUFJLEtBQUssQ0FBQywyRkFBMkYsS0FBSyxDQUFDLHVCQUF1QixHQUFHLENBQUMsQ0FBQztTQUM5STtRQUVELE1BQU0sT0FBTyxTQUFHLEtBQUssQ0FBQyxjQUFjLG1DQUFJLDJCQUEyQixDQUFDLElBQUksRUFBRSxDQUFDO1FBQzNFLE1BQU0sT0FBTyxTQUFHLEtBQUssQ0FBQyxjQUFjLG1DQUFJLDJCQUEyQixDQUFDLElBQUksRUFBRSxDQUFDO1FBQzNFLE1BQU0sWUFBWSxTQUFHLEtBQUssQ0FBQyxtQkFBbUIsbUNBQUksZ0NBQWdDLENBQUMsSUFBSSxFQUFFLENBQUM7UUFFMUYsTUFBTSxRQUFRLEdBQUcsSUFBSSw2Q0FBc0IsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzVELHlCQUF5QixFQUFFO2dCQUN6QixJQUFJLEVBQUUsdUJBQXVCO2dCQUM3QixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87Z0JBQ3RCLGFBQWEsRUFBRTtvQkFDYixjQUFjLEVBQUUsT0FBTyxDQUFDLFFBQVE7b0JBQ2hDLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTztpQkFDekI7Z0JBQ0QsYUFBYSxFQUFFO29CQUNiLGNBQWMsRUFBRSxPQUFPLENBQUMsUUFBUTtvQkFDaEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPO2lCQUN6QjtnQkFDRCxrQkFBa0IsRUFBRTtvQkFDbEIsbUJBQW1CLEVBQUUsWUFBWSxDQUFDLFFBQVE7b0JBQzFDLFlBQVksRUFBRSxZQUFZLENBQUMsWUFBWTtpQkFDeEM7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxxQkFBcUIsR0FBRyxRQUFRLENBQUMsR0FBRyxDQUFDO0lBQzVDLENBQUM7Ozs7OztJQWpETSxNQUFNLENBQUMseUJBQXlCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUscUJBQTZCO1FBQ2pHLE9BQU8sSUFBSSxLQUFNLFNBQVEsZUFBUTtZQUF0Qjs7Z0JBQ08sMEJBQXFCLEdBQUcscUJBQXFCLENBQUM7WUFDaEUsQ0FBQztTQUFBLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ2YsQ0FBQztJQUVELHFEQUFxRDtJQUM3QyxNQUFNLENBQUMsOEJBQThCLENBQUMsNEJBQW9DO1FBQ2hGLE9BQU8sSUFBSTtZQUFBO2dCQUNPLDBCQUFxQixHQUFHLDRCQUE0QixDQUFDO1lBQ3ZFLENBQUM7U0FBQSxFQUFFLENBQUM7SUFDTixDQUFDOztBQXpCSCxrREFnRUM7Ozs7Ozs7O0FBN0R3Qiw4Q0FBMEIsR0FBRyxtQkFBbUIsQ0FBQyw4QkFBOEIsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDOzs7Ozs7QUFFeEgsc0NBQWtCLEdBQUcsbUJBQW1CLENBQUMsOEJBQThCLENBQUMsc0NBQXNDLENBQUMsQ0FBQzs7Ozs7O0FBRWhILGtDQUFjLEdBQUcsbUJBQW1CLENBQUMsOEJBQThCLENBQUMsc0NBQXNDLENBQUMsQ0FBQzs7Ozs7O0FBRTVHLDhCQUFVLEdBQUcsbUJBQW1CLENBQUMsOEJBQThCLENBQUMsc0NBQXNDLENBQUMsQ0FBQzs7Ozs7O0FBRXhHLDBDQUFzQixHQUFHLG1CQUFtQixDQUFDLDhCQUE4QixDQUFDLHNDQUFzQyxDQUFDLENBQUM7Ozs7OztBQTREN0ksTUFBYSwyQkFBMkI7SUF1QnRDLFlBQW9CLFFBQWdCLEVBQUUsT0FBa0I7UUFDdEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUM7UUFDekIsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7SUFDekIsQ0FBQzs7Ozs7Ozs7SUFyQk0sTUFBTSxDQUFDLElBQUksS0FBSyxPQUFPLElBQUksMkJBQTJCLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7Ozs7SUFHakUsTUFBTSxDQUFDLEdBQUcsS0FBSyxPQUFPLElBQUksMkJBQTJCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDOzs7Ozs7SUFHL0QsTUFBTSxDQUFDLFNBQVMsQ0FBQyxHQUFHLE9BQWlCO1FBQzFDLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO1NBQ2xFO1FBQ0QsT0FBTyxJQUFJLDJCQUEyQixDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMvRCxDQUFDO0NBV0Y7QUEzQkQsa0VBMkJDOzs7Ozs7QUFNRCxNQUFhLDJCQUEyQjtJQW9DdEMsWUFBb0IsUUFBZ0IsRUFBRSxPQUFrQjtRQUN0RCxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQztRQUN6QixJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQztJQUN6QixDQUFDOzs7Ozs7OztJQWxDTSxNQUFNLENBQUMsSUFBSSxLQUFLLE9BQU8sSUFBSSwyQkFBMkIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7Ozs7OztJQU9qRSxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsaUJBQTJCO1FBQzlDLElBQUksaUJBQWlCLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNoQyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsQ0FBQyxFQUFFO2dCQUN4RSxNQUFNLElBQUksS0FBSyxDQUFDLDZHQUE2RyxDQUFDLENBQUM7YUFDaEk7WUFDRCxPQUFPLElBQUksMkJBQTJCLENBQUMsaUNBQWlDLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztTQUM5RjthQUFNO1lBQ0wsT0FBTyxJQUFJLDJCQUEyQixDQUFDLFdBQVcsQ0FBQyxDQUFDO1NBQ3JEO0lBQ0gsQ0FBQzs7Ozs7O0lBR00sTUFBTSxDQUFDLFNBQVMsQ0FBQyxHQUFHLE9BQWlCO1FBQzFDLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsQ0FBQyxDQUFDO1NBQ2xFO1FBQ0QsT0FBTyxJQUFJLDJCQUEyQixDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUMvRCxDQUFDO0NBV0Y7QUF4Q0Qsa0VBd0NDOzs7Ozs7QUFPRCxNQUFhLGdDQUFnQztJQXVCM0MsWUFBb0IsUUFBZ0IsRUFBRSxZQUF1QjtRQUMzRCxJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsQ0FBQztRQUN6QixJQUFJLENBQUMsWUFBWSxHQUFHLFlBQVksQ0FBQztJQUNuQyxDQUFDOzs7Ozs7OztJQXJCTSxNQUFNLENBQUMsSUFBSSxLQUFLLE9BQU8sSUFBSSxnQ0FBZ0MsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7OztJQUd0RSxNQUFNLENBQUMsR0FBRyxLQUFLLE9BQU8sSUFBSSxnQ0FBZ0MsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7OztJQUdwRSxNQUFNLENBQUMsU0FBUyxDQUFDLEdBQUcsWUFBc0I7UUFDL0MsSUFBSSxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7U0FDeEU7UUFDRCxPQUFPLElBQUksZ0NBQWdDLENBQUMsV0FBVyxFQUFFLFlBQVksQ0FBQyxDQUFDO0lBQ3pFLENBQUM7Q0FXRjtBQTNCRCw0RUEyQkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBOYW1lcywgUmVzb3VyY2UsIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENmbk9yaWdpblJlcXVlc3RQb2xpY3kgfSBmcm9tICcuL2Nsb3VkZnJvbnQuZ2VuZXJhdGVkJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSU9yaWdpblJlcXVlc3RQb2xpY3kge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgb3JpZ2luUmVxdWVzdFBvbGljeUlkOiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBPcmlnaW5SZXF1ZXN0UG9saWN5UHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgb3JpZ2luUmVxdWVzdFBvbGljeU5hbWU/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbW1lbnQ/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb29raWVCZWhhdmlvcj86IE9yaWdpblJlcXVlc3RDb29raWVCZWhhdmlvcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGhlYWRlckJlaGF2aW9yPzogT3JpZ2luUmVxdWVzdEhlYWRlckJlaGF2aW9yO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcXVlcnlTdHJpbmdCZWhhdmlvcj86IE9yaWdpblJlcXVlc3RRdWVyeVN0cmluZ0JlaGF2aW9yO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE9yaWdpblJlcXVlc3RQb2xpY3kgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElPcmlnaW5SZXF1ZXN0UG9saWN5IHtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBVU0VSX0FHRU5UX1JFRkVSRVJfSEVBREVSUyA9IE9yaWdpblJlcXVlc3RQb2xpY3kuZnJvbU1hbmFnZWRPcmlnaW5SZXF1ZXN0UG9saWN5KCdhY2JhNDU5NS1iZDI4LTQ5YjgtYjlmZS0xMzMxN2MwMzkwZmEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENPUlNfQ1VTVE9NX09SSUdJTiA9IE9yaWdpblJlcXVlc3RQb2xpY3kuZnJvbU1hbmFnZWRPcmlnaW5SZXF1ZXN0UG9saWN5KCc1OTc4MWE1Yi0zOTAzLTQxZjMtYWZjYi1hZjYyOTI5Y2NkZTEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDT1JTX1MzX09SSUdJTiA9IE9yaWdpblJlcXVlc3RQb2xpY3kuZnJvbU1hbmFnZWRPcmlnaW5SZXF1ZXN0UG9saWN5KCc4OGE1ZWFmNC0yZmQ0LTQ3MDktYjM3MC1iNGM2NTBlYTNmY2YnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgQUxMX1ZJRVdFUiA9IE9yaWdpblJlcXVlc3RQb2xpY3kuZnJvbU1hbmFnZWRPcmlnaW5SZXF1ZXN0UG9saWN5KCcyMTZhZGVmNi01YzdmLTQ3ZTQtYjk4OS01NDkyZWFmYTA3ZDMnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IEVMRU1FTlRBTF9NRURJQV9UQUlMT1IgPSBPcmlnaW5SZXF1ZXN0UG9saWN5LmZyb21NYW5hZ2VkT3JpZ2luUmVxdWVzdFBvbGljeSgnNzc1MTMzYmMtMTVmMi00OWY5LWFiZWEtYWZiMmUwYmY2N2QyJyk7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZnJvbU9yaWdpblJlcXVlc3RQb2xpY3lJZChzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBvcmlnaW5SZXF1ZXN0UG9saWN5SWQ6IHN0cmluZyk6IElPcmlnaW5SZXF1ZXN0UG9saWN5IHtcbiAgICByZXR1cm4gbmV3IGNsYXNzIGV4dGVuZHMgUmVzb3VyY2UgaW1wbGVtZW50cyBJT3JpZ2luUmVxdWVzdFBvbGljeSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgb3JpZ2luUmVxdWVzdFBvbGljeUlkID0gb3JpZ2luUmVxdWVzdFBvbGljeUlkO1xuICAgIH0oc2NvcGUsIGlkKTtcbiAgfVxuXG4gIC8qKiBVc2UgYW4gZXhpc3RpbmcgbWFuYWdlZCBvcmlnaW4gcmVxdWVzdCBwb2xpY3kuICovXG4gIHByaXZhdGUgc3RhdGljIGZyb21NYW5hZ2VkT3JpZ2luUmVxdWVzdFBvbGljeShtYW5hZ2VkT3JpZ2luUmVxdWVzdFBvbGljeUlkOiBzdHJpbmcpOiBJT3JpZ2luUmVxdWVzdFBvbGljeSB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBpbXBsZW1lbnRzIElPcmlnaW5SZXF1ZXN0UG9saWN5IHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBvcmlnaW5SZXF1ZXN0UG9saWN5SWQgPSBtYW5hZ2VkT3JpZ2luUmVxdWVzdFBvbGljeUlkO1xuICAgIH0oKTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBvcmlnaW5SZXF1ZXN0UG9saWN5SWQ6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogT3JpZ2luUmVxdWVzdFBvbGljeVByb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMub3JpZ2luUmVxdWVzdFBvbGljeU5hbWUsXG4gICAgfSk7XG5cbiAgICBjb25zdCBvcmlnaW5SZXF1ZXN0UG9saWN5TmFtZSA9IHByb3BzLm9yaWdpblJlcXVlc3RQb2xpY3lOYW1lID8/IE5hbWVzLnVuaXF1ZUlkKHRoaXMpO1xuICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKG9yaWdpblJlcXVlc3RQb2xpY3lOYW1lKSAmJiAhb3JpZ2luUmVxdWVzdFBvbGljeU5hbWUubWF0Y2goL15bXFx3LV0rJC9pKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGAnb3JpZ2luUmVxdWVzdFBvbGljeU5hbWUnIGNhbiBvbmx5IGluY2x1ZGUgJy0nLCAnXycsIGFuZCBhbHBoYW51bWVyaWMgY2hhcmFjdGVycywgZ290OiAnJHtwcm9wcy5vcmlnaW5SZXF1ZXN0UG9saWN5TmFtZX0nYCk7XG4gICAgfVxuXG4gICAgY29uc3QgY29va2llcyA9IHByb3BzLmNvb2tpZUJlaGF2aW9yID8/IE9yaWdpblJlcXVlc3RDb29raWVCZWhhdmlvci5ub25lKCk7XG4gICAgY29uc3QgaGVhZGVycyA9IHByb3BzLmhlYWRlckJlaGF2aW9yID8/IE9yaWdpblJlcXVlc3RIZWFkZXJCZWhhdmlvci5ub25lKCk7XG4gICAgY29uc3QgcXVlcnlTdHJpbmdzID0gcHJvcHMucXVlcnlTdHJpbmdCZWhhdmlvciA/PyBPcmlnaW5SZXF1ZXN0UXVlcnlTdHJpbmdCZWhhdmlvci5ub25lKCk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5PcmlnaW5SZXF1ZXN0UG9saWN5KHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIG9yaWdpblJlcXVlc3RQb2xpY3lDb25maWc6IHtcbiAgICAgICAgbmFtZTogb3JpZ2luUmVxdWVzdFBvbGljeU5hbWUsXG4gICAgICAgIGNvbW1lbnQ6IHByb3BzLmNvbW1lbnQsXG4gICAgICAgIGNvb2tpZXNDb25maWc6IHtcbiAgICAgICAgICBjb29raWVCZWhhdmlvcjogY29va2llcy5iZWhhdmlvcixcbiAgICAgICAgICBjb29raWVzOiBjb29raWVzLmNvb2tpZXMsXG4gICAgICAgIH0sXG4gICAgICAgIGhlYWRlcnNDb25maWc6IHtcbiAgICAgICAgICBoZWFkZXJCZWhhdmlvcjogaGVhZGVycy5iZWhhdmlvcixcbiAgICAgICAgICBoZWFkZXJzOiBoZWFkZXJzLmhlYWRlcnMsXG4gICAgICAgIH0sXG4gICAgICAgIHF1ZXJ5U3RyaW5nc0NvbmZpZzoge1xuICAgICAgICAgIHF1ZXJ5U3RyaW5nQmVoYXZpb3I6IHF1ZXJ5U3RyaW5ncy5iZWhhdmlvcixcbiAgICAgICAgICBxdWVyeVN0cmluZ3M6IHF1ZXJ5U3RyaW5ncy5xdWVyeVN0cmluZ3MsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy5vcmlnaW5SZXF1ZXN0UG9saWN5SWQgPSByZXNvdXJjZS5yZWY7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBPcmlnaW5SZXF1ZXN0Q29va2llQmVoYXZpb3Ige1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbm9uZSgpIHsgcmV0dXJuIG5ldyBPcmlnaW5SZXF1ZXN0Q29va2llQmVoYXZpb3IoJ25vbmUnKTsgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbCgpIHsgcmV0dXJuIG5ldyBPcmlnaW5SZXF1ZXN0Q29va2llQmVoYXZpb3IoJ2FsbCcpOyB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgYWxsb3dMaXN0KC4uLmNvb2tpZXM6IHN0cmluZ1tdKSB7XG4gICAgaWYgKGNvb2tpZXMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0F0IGxlYXN0IG9uZSBjb29raWUgdG8gYWxsb3cgbXVzdCBiZSBwcm92aWRlZCcpO1xuICAgIH1cbiAgICByZXR1cm4gbmV3IE9yaWdpblJlcXVlc3RDb29raWVCZWhhdmlvcignd2hpdGVsaXN0JywgY29va2llcyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgYmVoYXZpb3I6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGNvb2tpZXM/OiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGJlaGF2aW9yOiBzdHJpbmcsIGNvb2tpZXM/OiBzdHJpbmdbXSkge1xuICAgIHRoaXMuYmVoYXZpb3IgPSBiZWhhdmlvcjtcbiAgICB0aGlzLmNvb2tpZXMgPSBjb29raWVzO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgT3JpZ2luUmVxdWVzdEhlYWRlckJlaGF2aW9yIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBub25lKCkgeyByZXR1cm4gbmV3IE9yaWdpblJlcXVlc3RIZWFkZXJCZWhhdmlvcignbm9uZScpOyB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbCguLi5jbG91ZGZyb250SGVhZGVyczogc3RyaW5nW10pIHtcbiAgICBpZiAoY2xvdWRmcm9udEhlYWRlcnMubGVuZ3RoID4gMCkge1xuICAgICAgaWYgKCFjbG91ZGZyb250SGVhZGVycy5ldmVyeShoZWFkZXIgPT4gaGVhZGVyLnN0YXJ0c1dpdGgoJ0Nsb3VkRnJvbnQtJykpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignYWRkaXRpb25hbCBDbG91ZEZyb250IGhlYWRlcnMgcGFzc2VkIHRvIGBPcmlnaW5SZXF1ZXN0SGVhZGVyQmVoYXZpb3IuYWxsKClgIG11c3QgYmVnaW4gd2l0aCBcXCdDbG91ZEZyb250LVxcJycpO1xuICAgICAgfVxuICAgICAgcmV0dXJuIG5ldyBPcmlnaW5SZXF1ZXN0SGVhZGVyQmVoYXZpb3IoJ2FsbFZpZXdlckFuZFdoaXRlbGlzdENsb3VkRnJvbnQnLCBjbG91ZGZyb250SGVhZGVycyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHJldHVybiBuZXcgT3JpZ2luUmVxdWVzdEhlYWRlckJlaGF2aW9yKCdhbGxWaWV3ZXInKTtcbiAgICB9XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbG93TGlzdCguLi5oZWFkZXJzOiBzdHJpbmdbXSkge1xuICAgIGlmIChoZWFkZXJzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgaGVhZGVyIHRvIGFsbG93IG11c3QgYmUgcHJvdmlkZWQnKTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBPcmlnaW5SZXF1ZXN0SGVhZGVyQmVoYXZpb3IoJ3doaXRlbGlzdCcsIGhlYWRlcnMpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGJlaGF2aW9yOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGhlYWRlcnM/OiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGJlaGF2aW9yOiBzdHJpbmcsIGhlYWRlcnM/OiBzdHJpbmdbXSkge1xuICAgIHRoaXMuYmVoYXZpb3IgPSBiZWhhdmlvcjtcbiAgICB0aGlzLmhlYWRlcnMgPSBoZWFkZXJzO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgT3JpZ2luUmVxdWVzdFF1ZXJ5U3RyaW5nQmVoYXZpb3Ige1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbm9uZSgpIHsgcmV0dXJuIG5ldyBPcmlnaW5SZXF1ZXN0UXVlcnlTdHJpbmdCZWhhdmlvcignbm9uZScpOyB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgYWxsKCkgeyByZXR1cm4gbmV3IE9yaWdpblJlcXVlc3RRdWVyeVN0cmluZ0JlaGF2aW9yKCdhbGwnKTsgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgYWxsb3dMaXN0KC4uLnF1ZXJ5U3RyaW5nczogc3RyaW5nW10pIHtcbiAgICBpZiAocXVlcnlTdHJpbmdzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgcXVlcnkgc3RyaW5nIHRvIGFsbG93IG11c3QgYmUgcHJvdmlkZWQnKTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBPcmlnaW5SZXF1ZXN0UXVlcnlTdHJpbmdCZWhhdmlvcignd2hpdGVsaXN0JywgcXVlcnlTdHJpbmdzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGJlaGF2aW9yOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBxdWVyeVN0cmluZ3M/OiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGJlaGF2aW9yOiBzdHJpbmcsIHF1ZXJ5U3RyaW5ncz86IHN0cmluZ1tdKSB7XG4gICAgdGhpcy5iZWhhdmlvciA9IGJlaGF2aW9yO1xuICAgIHRoaXMucXVlcnlTdHJpbmdzID0gcXVlcnlTdHJpbmdzO1xuICB9XG59XG4iXX0=