# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['pygenomeviz', 'pygenomeviz.parser', 'pygenomeviz.scripts']

package_data = \
{'': ['*'],
 'pygenomeviz': ['viewer/*',
                 'viewer/assets/*',
                 'viewer/assets/lib/*',
                 'viewer/assets/lib/images/*']}

install_requires = \
['biopython>=1.79,<2.0',
 'matplotlib>=3.5.2,<4.0.0',
 'numpy>=1.21,<2.0',
 'typing-extensions>=4.3.0,<5.0.0']

entry_points = \
{'console_scripts': ['pgv-download-dataset = '
                     'pygenomeviz.scripts.download_dataset:main',
                     'pgv-mmseqs = pygenomeviz.scripts.mmseqs:main',
                     'pgv-mummer = pygenomeviz.scripts.mummer:main',
                     'pgv-pmauve = pygenomeviz.scripts.pmauve:main',
                     'pgv-simpleplot = pygenomeviz.scripts.simpleplot:main']}

setup_kwargs = {
    'name': 'pygenomeviz',
    'version': '0.3.0',
    'description': 'A genome visualization python package for comparative genomics',
    'long_description': '# pyGenomeViz\n\n![Python3](https://img.shields.io/badge/Language-Python3-steelblue)\n![OS](https://img.shields.io/badge/OS-_Windows_|_Mac_|_Linux-steelblue)\n![License](https://img.shields.io/badge/License-MIT-steelblue)\n[![Latest PyPI version](https://img.shields.io/pypi/v/pygenomeviz.svg)](https://pypi.python.org/pypi/pygenomeviz)\n[![Bioconda](https://img.shields.io/conda/vn/bioconda/pygenomeviz.svg?color=green)](https://anaconda.org/bioconda/pygenomeviz)\n[![CI](https://github.com/moshi4/pyGenomeViz/actions/workflows/ci.yml/badge.svg)](https://github.com/moshi4/pyGenomeViz/actions/workflows/ci.yml)\n\n## Table of contents\n\n- [Overview](#overview)\n- [Installation](#installation)\n- [API Examples](#api-examples)\n- [CLI Examples](#cli-examples)\n- [Interactive HTML Viewer](#interactive-html-viewer)\n- [Inspiration](#inspiration)\n\n## Overview\n\npyGenomeViz is a genome visualization python package for comparative genomics implemented based on matplotlib.\nThis package is developed for the purpose of easily and beautifully plotting genomic\nfeatures and sequence similarity comparison links between multiple genomes.\nIt supports genome visualization of Genbank/GFF format file and can be saved figure in various formats (JPG/PNG/SVG/PDF/HTML).\nUser can use pyGenomeViz for interactive genome visualization figure plotting on jupyter notebook,\nor automatic genome visualization figure plotting in genome analysis scripts/pipelines.\n\nFor more information, please see full documentation [here](https://moshi4.github.io/pyGenomeViz/).\n\n![pygenomeviz_gallery.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/pygenomeviz_gallery.png)  \n**Fig.1 pyGenomeViz example plot gallery**\n\n## Installation\n\n`Python 3.7 or later` is required for installation.\n\n**Install PyPI package:**\n\n    pip install pygenomeviz\n\n**Install bioconda package:**\n\n    conda install -c conda-forge -c bioconda pygenomeviz\n\n**Use Docker ([Image Registry](https://github.com/moshi4/pyGenomeViz/pkgs/container/pygenomeviz)):**\n\n    docker pull ghcr.io/moshi4/pygenomeviz:latest\n    docker run -it --rm ghcr.io/moshi4/pygenomeviz:latest pgv-pmauve -h\n\n## API Examples\n\nJupyter notebooks containing code examples below is available [here](https://moshi4.github.io/pyGenomeViz/getting_started/).\n\n### Basic Example\n\n#### Single Track\n\n```python\nfrom pygenomeviz import GenomeViz\n\nname, genome_size = "Tutorial 01", 5000\ncds_list = ((100, 900, -1), (1100, 1300, 1), (1350, 1500, 1), (1520, 1700, 1), (1900, 2200, -1), (2500, 2700, 1), (2700, 2800, -1), (2850, 3000, -1), (3100, 3500, 1), (3600, 3800, -1), (3900, 4200, -1), (4300, 4700, -1), (4800, 4850, 1))\n\ngv = GenomeViz()\ntrack = gv.add_feature_track(name, genome_size)\nfor idx, cds in enumerate(cds_list, 1):\n    start, end, strand = cds\n    track.add_feature(start, end, strand, label=f"CDS{idx:02d}")\n\ngv.savefig("example01.png")\n```\n\n![example01.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/example01.png)\n\n#### Multiple Tracks & Links\n\n```python\nfrom pygenomeviz import GenomeViz\n\ngenome_list = (\n    {"name": "genome 01", "size": 1000, "cds_list": ((150, 300, 1), (500, 700, -1), (750, 950, 1))},\n    {"name": "genome 02", "size": 1300, "cds_list": ((50, 200, 1), (350, 450, 1), (700, 900, -1), (950, 1150, -1))},\n    {"name": "genome 03", "size": 1200, "cds_list": ((150, 300, 1), (350, 450, -1), (500, 700, -1), (700, 900, -1))},\n)\n\ngv = GenomeViz(tick_style="axis")\nfor genome in genome_list:\n    name, size, cds_list = genome["name"], genome["size"], genome["cds_list"]\n    track = gv.add_feature_track(name, size)\n    for idx, cds in enumerate(cds_list, 1):\n        start, end, strand = cds\n        track.add_feature(start, end, strand, label=f"gene{idx:02d}", linewidth=1, labelrotation=0, labelvpos="top", labelhpos="center", labelha="center")\n\n# Add links between "genome 01" and "genome 02"\ngv.add_link(("genome 01", 150, 300), ("genome 02", 50, 200))\ngv.add_link(("genome 01", 700, 500), ("genome 02", 900, 700))\ngv.add_link(("genome 01", 750, 950), ("genome 02", 1150, 950))\n# Add links between "genome 02" and "genome 03"\ngv.add_link(("genome 02", 50, 200), ("genome 03", 150, 300), normal_color="skyblue", inverted_color="lime", curve=True)\ngv.add_link(("genome 02", 350, 450), ("genome 03", 450, 350), normal_color="skyblue", inverted_color="lime", curve=True)\ngv.add_link(("genome 02", 900, 700), ("genome 03", 700, 500), normal_color="skyblue", inverted_color="lime", curve=True)\ngv.add_link(("genome 03", 900, 700), ("genome 02", 1150, 950), normal_color="skyblue", inverted_color="lime", curve=True)\n\ngv.savefig("example02.png")\n```\n\n![example02.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/example02.png)\n\n#### Exon Features\n\n```python\nfrom pygenomeviz import GenomeViz\n\nexon_regions1 = [(0, 210), (300, 480), (590, 800), (850, 1000), (1030, 1300)]\nexon_regions2 = [(1500, 1710), (2000, 2480), (2590, 2800)]\nexon_regions3 = [(3000, 3300), (3400, 3690), (3800, 4100), (4200, 4620)]\n\ngv = GenomeViz()\ntrack = gv.add_feature_track(name=f"Exon Features", size=5000)\ntrack.add_exon_feature(exon_regions1, strand=1, plotstyle="box", label="box", labelrotation=0, labelha="center")\ntrack.add_exon_feature(exon_regions2, strand=-1, plotstyle="arrow", label="arrow", labelrotation=0, labelha="center", facecolor="darkgrey", intron_patch_kws={"ec": "red"})\n\nexon_labels = [f"exon{i+1}" for i in range(len(exon_regions3))]\ntrack.add_exon_feature(exon_regions3, strand=1, plotstyle="bigarrow", label="bigarrow", facecolor="lime", linewidth=1, exon_labels=exon_labels, labelrotation=0, labelha="center", exon_label_kws={"y": 0, "va": "center", "color": "blue"})\n\ngv.savefig("example03.png")\n```\n\n![example03.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/example03.png)\n\n### Practical Example\n\n#### Add Features from Genbank file\n\n```python\nfrom pygenomeviz import Genbank, GenomeViz, load_dataset\n\ngbk_files, _ = load_dataset("enterobacteria_phage")\ngbk = Genbank(gbk_files[0])\n\ngv = GenomeViz()\ntrack = gv.add_feature_track(gbk.name, gbk.range_size)\ntrack.add_genbank_features(gbk)\n\ngv.savefig("example04.png")\n```\n\n![example04.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/example04.png)\n\n#### Add Features from GFF file\n\n```python\nfrom pygenomeviz import Gff, GenomeViz, load_example_gff\n\ngff_file = load_example_gff("enterobacteria_phage.gff")\ngff = Gff(gff_file, min_range=5000, max_range=25000)\n\ngv = GenomeViz(fig_track_height=0.7, tick_track_ratio=0.5, tick_style="bar")\ntrack = gv.add_feature_track(gff.name, size=gff.range_size, start_pos=gff.min_range)\ntrack.add_gff_features(gff, plotstyle="arrow", facecolor="tomato")\ntrack.set_sublabel()\n\ngv.savefig("example05.png")\n```\n\n![example05.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/example05.png)\n\n#### Multiple Tracks & Links from Genbank files\n\n```python\nfrom pygenomeviz import Genbank, GenomeViz, load_dataset\n\ngv = GenomeViz(\n    fig_track_height=0.7,\n    feature_track_ratio=0.2,\n    tick_track_ratio=0.4,\n    tick_style="bar",\n    align_type="center",\n)\n\ngbk_files, links = load_dataset("escherichia_phage")\nfor gbk_file in gbk_files:\n    gbk = Genbank(gbk_file)\n    track = gv.add_feature_track(gbk.name, gbk.range_size)\n    track.add_genbank_features(gbk, facecolor="limegreen", linewidth=0.5, arrow_shaft_ratio=1.0)\n\nfor link in links:\n    link_data1 = (link.ref_name, link.ref_start, link.ref_end)\n    link_data2 = (link.query_name, link.query_start, link.query_end)\n    gv.add_link(link_data1, link_data2, v=link.identity, curve=True)\n\ngv.savefig("example06.png")\n```\n\n![example06.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/example06.png)\n\n### Customization Tips\n\nSince pyGenomeViz is implemented based on matplotlib, users can easily customize\nthe figure in the manner of matplotlib. Here are some tips for figure customization.\n\n#### Customization Tips 01\n\n- Add `GC Content` & `GC skew` subtrack\n- Add annotation label & fillbox\n- Add colorbar for links identity\n\n<details>\n<summary>Code</summary>\n\n```python\nfrom pygenomeviz import Genbank, GenomeViz, load_dataset\n\ngv = GenomeViz(\n    fig_width=12,\n    fig_track_height=0.7,\n    feature_track_ratio=0.5,\n    tick_track_ratio=0.3,\n    tick_style="axis",\n    tick_labelsize=10,\n)\n\ngbk_files, links = load_dataset("erwinia_phage")\ngbk_list = [Genbank(gbk_file) for gbk_file in gbk_files]\nfor gbk in gbk_list:\n    track = gv.add_feature_track(gbk.name, gbk.range_size, labelsize=15)\n    track.add_genbank_features(gbk, plotstyle="arrow")\n\nmin_identity = int(min(link.identity for link in links))\nfor link in links:\n    link_data1 = (link.ref_name, link.ref_start, link.ref_end)\n    link_data2 = (link.query_name, link.query_start, link.query_end)\n    gv.add_link(link_data1, link_data2, v=link.identity, vmin=min_identity)\n\n# Add subtracks to top track for plotting \'GC content\' & \'GC skew\'\ngv.top_track.add_subtrack(ratio=0.7, name="gc_content")\ngv.top_track.add_subtrack(ratio=0.7, name="gc_skew")\n\nfig = gv.plotfig()\n\n# Add label annotation to top track\ntop_track = gv.top_track  # or, gv.get_track("MT939486") or gv.get_tracks()[0]\nlabel, start, end = "Inverted", 310000 + top_track.offset, 358000 + top_track.offset\ncenter = int((start + end) / 2)\ntop_track.ax.hlines(1.5, start, end, colors="red", linewidth=1, linestyles="dashed", clip_on=False)\ntop_track.ax.text(center, 2.0, label, fontsize=12, color="red", ha="center", va="bottom")\n\n# Add fillbox to top track\nx, y = (start, start, end, end), (1, -1, -1, 1)\ntop_track.ax.fill(x, y, fc="lime", linewidth=0, alpha=0.1, zorder=-10)\n\n# Plot GC content for top track\npos_list, gc_content_list = gbk_list[0].calc_gc_content()\npos_list += gv.top_track.offset  # Offset is required if align_type is not \'left\'\ngc_content_ax = gv.top_track.subtracks[0].ax\ngc_content_ax.set_ylim(bottom=0, top=max(gc_content_list))\ngc_content_ax.fill_between(pos_list, gc_content_list, alpha=0.2, color="blue")\ngc_content_ax.text(gv.top_track.offset, max(gc_content_list) / 2, "GC(%) ", ha="right", va="center", color="blue")\n\n# Plot GC skew for top track\npos_list, gc_skew_list = gbk_list[0].calc_gc_skew()\npos_list += gv.top_track.offset  # Offset is required if align_type is not \'left\'\ngc_skew_abs_max = max(abs(gc_skew_list))\ngc_skew_ax = gv.top_track.subtracks[1].ax\ngc_skew_ax.set_ylim(bottom=-gc_skew_abs_max, top=gc_skew_abs_max)\ngc_skew_ax.fill_between(pos_list, gc_skew_list, alpha=0.2, color="red")\ngc_skew_ax.text(gv.top_track.offset, 0, "GC skew ", ha="right", va="center", color="red")\n\n# Set coloarbar for link\ngv.set_colorbar(fig, vmin=min_identity)\n\nfig.savefig("example07.png")\n```\n\n</details>\n\n![example07.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/example07.png)\n\n#### Customization Tips 02\n\n- Add legends\n- Add colorbar for links identity\n\n<details>\n<summary>Code</summary>\n\n```python\nfrom matplotlib.lines import Line2D\nfrom matplotlib.patches import Patch\n\nfrom pygenomeviz import Genbank, GenomeViz, load_dataset\n\ngv = GenomeViz(\n    fig_width=10,\n    fig_track_height=0.5,\n    feature_track_ratio=0.5,\n    tick_track_ratio=0.3,\n    align_type="center",\n    tick_style="bar",\n    tick_labelsize=10,\n)\n\ngbk_files, links = load_dataset("enterobacteria_phage")\nfor idx, gbk_file in enumerate(gbk_files):\n    gbk = Genbank(gbk_file)\n    track = gv.add_feature_track(gbk.name, gbk.range_size, labelsize=10)\n    track.add_genbank_features(\n        gbk,\n        label_type="product" if idx == 0 else None,  # Labeling only top track\n        label_handle_func=lambda s: "" if s.startswith("hypothetical") else s,  # Ignore \'hypothetical ~~~\' label\n        labelsize=8,\n        labelvpos="top",\n        facecolor="skyblue",\n        linewidth=0.5,\n    )\n\nnormal_color, inverted_color, alpha = "chocolate", "limegreen", 0.5\nmin_identity = int(min(link.identity for link in links))\nfor link in links:\n    link_data1 = (link.ref_name, link.ref_start, link.ref_end)\n    link_data2 = (link.query_name, link.query_start, link.query_end)\n    gv.add_link(link_data1, link_data2, normal_color, inverted_color, alpha, v=link.identity, vmin=min_identity, curve=True)\n\nfig = gv.plotfig()\n\n# Add Legends (Maybe there is a better way)\nhandles = [\n    Line2D([], [], marker=">", color="skyblue", label="CDS", ms=10, ls="none"),\n    Patch(color=normal_color, label="Normal Link"),\n    Patch(color=inverted_color, label="Inverted Link"),\n]\nfig.legend(handles=handles, bbox_to_anchor=(1, 1))\n\n# Set colorbar for link\ngv.set_colorbar(fig, bar_colors=[normal_color, inverted_color], alpha=alpha, vmin=min_identity, bar_label="Identity", bar_labelsize=10)\n\nfig.savefig("example08.png")\n```\n\n</details>\n\n![example08.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/example08.png)\n\n## CLI Examples\n\npyGenomeViz provides CLI workflow for visualization of genome alignment or\nreciprocal best-hit CDS search results with `MUMmer` or `MMseqs` or `progressiveMauve`.\nEach CLI workflow requires the installation of additional dependent tools to run.\n\n### MUMmer CLI Workflow Example\n\nSee [pgv-mummer document](https://moshi4.github.io/pyGenomeViz/cli-docs/pgv-mummer/) for details.\n\nDownload example dataset: `pgv-download-dataset -n erwinia_phage`\n\n> :warning: MUMmer must be installed in advance to run\n\n    pgv-mummer --gbk_resources MT939486.gbk MT939487.gbk MT939488.gbk LT960552.gbk \\\n               -o mummer_example --tick_style axis --align_type left --feature_plotstyle arrow\n\n![mummer_example.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/mummer_example1.png)  \n\n### MMseqs CLI Workflow Example\n\nSee [pgv-mmseqs document](https://moshi4.github.io/pyGenomeViz/cli-docs/pgv-mmseqs/) for details.\n\nDownload example dataset: `pgv-download-dataset -n enterobacteria_phage`\n\n> :warning: MMseqs must be installed in advance to run\n\n    pgv-mmseqs --gbk_resources NC_019724.gbk NC_024783.gbk NC_016566.gbk NC_013600.gbk NC_031081.gbk NC_028901.gbk \\\n               -o mmseqs_example --fig_track_height 0.7 --feature_linewidth 0.3 --tick_style bar --curve \\\n               --normal_link_color chocolate --inverted_link_color limegreen --feature_color skyblue\n\n![mmseqs_example.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/mmseqs_example3.png)  \n\n### progressiveMauve CLI Workflow Example\n\nSee [pgv-pmauve document](https://moshi4.github.io/pyGenomeViz/cli-docs/pgv-pmauve/) for details.\n\nDownload example dataset: `pgv-download-dataset -n escherichia_coli`\n\n> :warning: progressiveMauve must be installed in advance to run\n\n    pgv-pmauve --seq_files NC_000913.gbk NC_002695.gbk NC_011751.gbk NC_011750.gbk \\\n               -o pmauve_example --tick_style bar\n\n![pmauve_example.png](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/pmauve_example1.png)  \n\n## Interactive HTML Viewer\n\npyGenomeViz implements HTML file output functionality for interactive data visualization.\nIn API, HTML file can be output using `savefig_html` method. In CLI, user can select HTML file output option.\nAs shown below, data tooltip display, pan/zoom, object color change, text change, etc are available in HTML viewer\n([Demo Page](https://moshi4.github.io/pyGenomeViz/images/pgv-viewer-demo.html)).\n\n![pgv-viewer-demo.gif](https://raw.githubusercontent.com/moshi4/pyGenomeViz/main/docs/images/pgv-viewer-demo.gif)\n\n## Inspiration\n\npyGenomeViz was inspired by\n\n- [GenomeDiagram (BioPython)](https://github.com/biopython/biopython)\n- [Easyfig](http://mjsull.github.io/Easyfig/)\n- [genoplotR](https://genoplotr.r-forge.r-project.org/)\n- [gggenomes](https://github.com/thackl/gggenomes)\n',
    'author': 'moshi4',
    'author_email': 'None',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://moshi4.github.io/pyGenomeViz/',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7.1,<4.0.0',
}


setup(**setup_kwargs)
