# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['utility_bill_scraper',
 'utility_bill_scraper.bin',
 'utility_bill_scraper.canada.on']

package_data = \
{'': ['*']}

install_requires = \
['arrow>=1.2.0,<2.0.0',
 'beautifulsoup4>=4.6.0,<5.0.0',
 'google-api-python-client>=2.27.0,<3.0.0',
 'matplotlib>=3.2,<4.0',
 'numpy>=1.16,<2.0',
 'pandas>=1.1.0,<2.0.0',
 'pdfminer>=20191125,<20191126',
 'python-dotenv>=0.19.1,<0.20.0',
 'selenium>=3.141.0,<4.0.0']

entry_points = \
{'console_scripts': ['ubs = utility_bill_scraper.bin.ubs:main']}

setup_kwargs = {
    'name': 'utility-bill-scraper',
    'version': '0.10.2',
    'description': 'Utility bill scraper for extracting data from pdfs and websites.',
    'long_description': '# Utility bill scraper\n\n[![build](https://github.com/ryanfobel/utility-bill-scraper/actions/workflows/build.yml/badge.svg?branch=main)](https://github.com/ryanfobel/utility-bill-scraper/actions/workflows/build.yml)\n[![Binder](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/ryanfobel/utility-bill-scraper/main)\n[![PyPI version shields.io](https://img.shields.io/pypi/v/utility-bill-scraper.svg)](https://pypi.python.org/pypi/utility-bill-scraper/)\n\nDownload energy usage data and estimate CO<sub>2</sub> emissions from utility websites or pdf bills.\n\n## What is this?\n\nThe science is clear — global temperatures are rising and we need to drastically reduce our use of fossil fuels if we want to keep our planet habitable for future generations. Many governments around the world are declaring [climate emergencies](https://qz.com/1786781/which-cities-have-declared-climate-emergencies/) and are setting ambitious targets to reduce emissions (e.g., [net zero by 2050](https://www.ipcc.ch/sr15/), [50% reduction by 2030](https://www.npr.org/2021/04/16/987667828/how-the-u-s-could-halve-climate-emissions-by-2030)). While broad systemic changes are clearly required, individual action is also important. For those living in the [Global North](https://en.wikipedia.org/wiki/Global_North_and_Global_South), the majority of fossil-fuel emissions arise from heating/cooling our homes, using electricity, transportation, and the food we eat. It\'s obvious that we need to rapidly transition off fossil fuels, which will require (1) **clear targets**, (2) **a plan to achieve them**, and (3) **tools for measuring progress**.\n\nThere are [many](https://app.projectneutral.org/) [existing](https://coolclimate.berkeley.edu/calculator) [carbon](https://www.nature.org/en-us/get-involved/how-to-help/carbon-footprint-calculator/) [footprint](https://www.carbonfootprint.com/calculator.aspx) [calculators](https://www3.epa.gov/carbon-footprint-calculator/), but they often require manual data entry, leading most people to try them once to get a static snapshot at a point in time. While useful for gaining a high-level understanding of your personal emission sources, it would be much better if this footprint could be automatically updated over time to provide people with **feedback** on the impact of their actions. This project aims to do just that — to assist individuals with collecting data from utility companies (e.g., electricity and natural gas) by automatically downloading their data and converting usage into CO<sub>2</sub> emissions.\n\n![monthly_co2_emissions](https://raw.githubusercontent.com/ryanfobel/utility-bill-scraper/main/notebooks/canada/on/images/monthly_co2_emissions_natural_gas.png)\n\n<!-- START doctoc generated TOC please keep comment here to allow auto update -->\n<!-- DON\'T EDIT THIS SECTION, INSTEAD RE-RUN doctoc TO UPDATE -->\n## Table of contents\n\n- [Supported utilities](#supported-utilities)\n- [Install](#install)\n- [Data storage](#data-storage)\n- [Getting and plotting data using the Python API](#getting-and-plotting-data-using-the-python-api)\n  - [Update data](#update-data)\n  - [Plot monthly gas consumption](#plot-monthly-gas-consumption)\n  - [Convert gas consumption to CO<sub>2</sub> emissions](#convert-gas-consumption-to-cosub2sub-emissions)\n  - [Plot Annual CO<sub>2</sub> emissions](#plot-annual-cosub2sub-emissions)\n- [Command line utilities](#command-line-utilities)\n  - [Update data](#update-data-1)\n  - [Export data](#export-data)\n  - [Options](#options)\n  - [Environment variables](#environment-variables)\n- [Contributors](#contributors)\n\n<!-- END doctoc generated TOC please keep comment here to allow auto update -->\n\n## Supported utilities\n\nThe simplest way to get started is to click on one of the following links, which will open a session on https://mybinder.org where you can try downloading some data. **Note: after you click on the link, it will take a couple of minutes to load an interactive Jupyter notebook.** Then follow the instructions (e.g., provide your `username` and `password`) to run the notebook directly from your browser.\n\n * [Kitchener Utilities (gas & water)](https://mybinder.org/v2/gh/ryanfobel/utility-bill-scraper/main?labpath=notebooks%2Fcanada%2Fon%2Fkitchener_utilities.ipynb)\n * [Kitchener-Wilmot Hydro](https://mybinder.org/v2/gh/ryanfobel/utility-bill-scraper/main?labpath=notebooks%2Fcanada%2Fon%2Fkitchener_wilmot_hydro.ipynb)\n \n## Install\n\n```sh\npip install utility-bill-scraper\n```\n\n## Data storage\n\nAll data is stored in a file located at `$DATA_PATH/$UTILITY_NAME/monthly.csv`. The path to this file can be set as input argument when initializing an API object via the `data_path` argument.\n\n```\n└───data\n    └───Kitchener Utilities\n        └───monthly.csv\n        └───statements\n            │───2021-10-18 - Kitchener Utilities - $102.30.pdf\n            ...\n            └───2021-06-15 - Kitchener Utilities - $84.51.pdf\n```\n\n## Getting and plotting data using the Python API\n\n### Update data\n\n```python\nimport utility_bill_scraper.canada.on.kitchener_utilities as ku\n\napi = ku.KitchenerUtilitiesAPI(username=\'username\', password=\'password\')\n\n# Get new statements.\nupdates = api.update()\nif updates is not None:\n    print(f"{ len(updates) } statements_downloaded")\napi.history("monthly").tail()\n```\n![history tail](https://raw.githubusercontent.com/ryanfobel/utility-bill-scraper/main/notebooks/canada/on/images/history_tail.png)\n\n\n\n\n### Plot monthly gas consumption\n\n```python\nimport matplotlib.pyplot as plt\n\ndf = ku_api.history("monthly")\n\nplt.figure()\nplt.bar(df.index, df["Gas Consumption"], width=0.9, alpha=0.5)\nplt.xticks(rotation=90)\nplt.title("Monthly Gas Consumption")\nplt.ylabel("m$^3$")\n```\n\n![monthly gas consumption](https://raw.githubusercontent.com/ryanfobel/utility-bill-scraper/main/notebooks/canada/on/images/monthly_gas_consumption.png)\n\n### Convert gas consumption to CO<sub>2</sub> emissions\n\n```python\nfrom utility_bill_scraper import GAS_KGCO2_PER_CUBIC_METER\n\ndf["kgCO2"] = df["Gas Consumption"] * GAS_KGCO2_PER_CUBIC_METER\n```\n\n### Plot Annual CO<sub>2</sub> emissions\n\n```python\nfrom utility_bill_scraper import GAS_KGCO2_PER_CUBIC_METER\n\ndf["kgCO2"] = df["Gas Consumption"] * GAS_KGCO2_PER_CUBIC_METER\ndf["year"] = [int(x[0:4]) for x in df.index]\ndf["month"] = [int(x[5:7]) for x in df.index]\n\nplt.figure()\ndf.groupby("year").sum()["Gas Consumption"].plot.bar(width=bin_width, alpha=alpha)\nplt.ylabel("m$^3$")\nylim = plt.ylim()\nax = plt.gca()\nax2 = ax.twinx()\nplt.ylabel("tCO$_2$e")\nplt.ylim([GAS_KGCO2_PER_CUBIC_METER * y / 1e3 for y in ylim])\nplt.title("Annual CO$_2$e emissions from natural gas")\n```\n\n![annual co2_emissions](https://raw.githubusercontent.com/ryanfobel/utility-bill-scraper/main/notebooks/canada/on/images/annual_co2_emissions_natural_gas.png)\n\n## Command line utilities\n\nUpdate and export your utility data from the command line.\n\n### Update data\n\n```sh\n> ubs --utilty-name "Kitchener Utilities" update --user $USER --password $PASSWORD\n```\n\n### Export data\n\n```sh\n> ubs --utilty-name "Kitchener Utilities" export --output monthly.csv\n```\n\n### Options\n\n```sh\n> ubs --help\nusage: ubs [-h] [-e ENV] [--data-path DATA_PATH] [--utility-name UTILITY_NAME]\n              [--google-sa-credentials GOOGLE_SA_CREDENTIALS]\n              {update,export} ...\n\nubs (Utility bill scraper)\n\noptional arguments:\n  -h, --help            show this help message and exit\n  -e ENV, --env ENV     path to .env file\n  --data-path DATA_PATH\n                        folder containing the data file and statements\n  --utility-name UTILITY_NAME\n                        name of the utility\n  --google-sa-credentials GOOGLE_SA_CREDENTIALS\n                        google service account credentials\n\nsubcommands:\n  {update,export}       available sub-commands\n```\n\n### Environment variables\n\nNote that many options can be set via environment variables (useful for continuous integration and/or working with containers). The following can be set in your shell or via a `.env` file passed using the `-e` option.\n\n```sh\nDATA_PATH="folder containing the data file and statements"\nUTILITY_NAME="name of the utility"\nGOOGLE_SA_CREDENTIALS="google service account credentials"\nUSER="username"\nPASSWORD="password"\nSAVE_STATEMENTS="save downloaded statements (default=True)"\nMAX_DOWNLOADS="maximum number of statements to download"\n```\n\n## Contributors\n\n* Ryan Fobel ([@ryanfobel](https://github.com/ryanfobel))\n',
    'author': 'Ryan Fobel',
    'author_email': 'ryan@fobel.net',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.7.1,<3.11',
}


setup(**setup_kwargs)
