"""Persistent, stale-free memoization decorators for Python."""

# This file is part of Cachier.
# https://github.com/shaypal5/cachier

# Licensed under the MIT license:
# http://www.opensource.org/licenses/MIT-license
# Copyright (c) 2016, Shay Palachy <shaypal5@gmail.com>

import os
from functools import wraps

import datetime
try:  # for asynchronous file uploads
    from concurrent.futures import ThreadPoolExecutor
except ImportError:  # we're in python 2.x
    import pip
    PACKAGES = [
        package.project_name
        for package
        in pip.get_installed_distributions()
    ]
    if 'futures' not in PACKAGES:
        pip.main(['install', 'futures'])
    from concurrent.futures import ThreadPoolExecutor

from .pickle_core import _PickleCore
from .mongo_core import _MongoCore


DEFAULT_MAX_WORKERS = 8


def _max_workers():
    try:
        return int(os.environ['CACHIER_MAX_WORKERS'])
    except KeyError:
        os.environ['CACHIER_MAX_WORKERS'] = str(DEFAULT_MAX_WORKERS)
        return DEFAULT_MAX_WORKERS


def _set_max_workets(max_workers):
    os.environ['CACHIER_MAX_WORKERS'] = str(max_workers)
    _get_executor(True)


def _get_executor(reset=False):
    if reset:
        _get_executor.executor = ThreadPoolExecutor(_max_workers())
    try:
        return _get_executor.executor
    except AttributeError:
        _get_executor.executor = ThreadPoolExecutor(_max_workers())
        return _get_executor.executor


def _function_thread(core, key, func, args, kwds):
    try:
        func_res = func(*args, **kwds)
        core.set_entry(key, func_res)
    except BaseException as exc:  # pylint: disable=W0703
        print(
            'Function call failed with the following exception:\n{}'.format(
                exc))


def _calc_entry(core, key, func, args, kwds):
    try:
        core.mark_entry_being_calculated(key)
        # _get_executor().submit(core.mark_entry_being_calculated, key)
        func_res = func(*args, **kwds)
        core.set_entry(key, func_res)
        # _get_executor().submit(core.set_entry, key, func_res)
        return func_res
    finally:
        core.mark_entry_not_calculated(key)



def cachier(stale_after=None, next_time=False, pickle_reload=True,
            mongetter=None):
    """A persistent, stale-free memoization decorator.

    The positional and keyword arguments to the wrapped function must be
    hashable (i.e. Python's immutable built-in objects, not mutable
    containers). Also, notice that since objects which are instances of
    user-defined classes are hashable but all compare unequal (their hash
    value is their id), equal objects across different sessions will not yield
    identical keys.

    Arguments
    ---------
    stale_after (optional) : datetime.timedelta
        The time delta afterwhich a cached result is considered stale. Calls
        made after the result goes stale will trigger a recalculation of the
        result, but whether a stale or fresh result will be returned is
        determined by the optional next_time argument.
    next_time (optional) : bool
        If set to True, a stale result will be returned when finding one, not
        waiting for the calculation of the fresh result to return. Defaults to
        False.
    pickle_reload (optional) : bool
        If set to True, in-memory cache will be reloaded on each cache read,
        enabling different threads to share cache. Should be set to False for
        faster reads in single-thread programs. Defaults to True.
    mongetter (optional) : callable
        A callable that takes no arguments and returns a pymongo.Collection
        object with writing permissions. If unset a local pickle cache is used
        instead.
    """
    # print('Inside the wrapper maker')
    # print('mongetter={}'.format(mongetter))
    # print('stale_after={}'.format(stale_after))
    # print('next_time={}'.format(next_time))

    if mongetter:
        core = _MongoCore(mongetter, stale_after, next_time)
    else:
        core = _PickleCore(  # pylint: disable=R0204
            stale_after, next_time, pickle_reload)

    def _cachier_decorator(func):
        core.set_func(func)

        @wraps(func)
        def func_wrapper(*args, **kwds):  # pylint: disable=C0111,R0911
            # print('Inside general wrapper for {}.'.format(func.__name__))
            overwrite_cache = kwds.pop('overwrite_cache', False)
            ignore_cache = kwds.pop('ignore_cache', False)
            verbose_cache = kwds.pop('verbose_cache', False)
            if ignore_cache:
                return func(*args, **kwds)
            key, entry = core.get_entry(args, kwds)
            if overwrite_cache:
                return _calc_entry(core, key, func, args, kwds)
            if entry is not None:  # pylint: disable=R0101
                if verbose_cache:
                    print('Entry found.')
                if entry.get('value', None) is not None:
                    if verbose_cache:
                        print('Cached result found.')
                    if stale_after:
                        now = datetime.datetime.now()
                        if now - entry['time'] > stale_after:
                            if verbose_cache:
                                print('But it is stale... :(')
                            if entry['being_calculated']:
                                if next_time:
                                    if verbose_cache:
                                        print('Returning stale.')
                                    return entry['value']  # return stale val
                                if verbose_cache:
                                    print('Already calc. Waiting on change.')
                                return core.wait_on_entry_calc(key)
                            if next_time:
                                if verbose_cache:
                                    print('Async calc and return stale')
                                try:
                                    core.mark_entry_being_calculated(key)
                                    _get_executor().submit(
                                        _function_thread, core, key, func,
                                        args, kwds)
                                finally:
                                    core.mark_entry_not_calculated(key)
                                return entry['value']
                            if verbose_cache:
                                print('Calling decorated function and waiting')
                            return _calc_entry(core, key, func, args, kwds)
                    if verbose_cache:
                        print('And it is fresh!')
                    return entry['value']
                if entry['being_calculated']:
                    if verbose_cache:
                        print('No value but being calculated. Waiting.')
                    return core.wait_on_entry_calc(key)
            if verbose_cache:
                print('No entry found. No current calc. Calling like a boss.')
            return _calc_entry(core, key, func, args, kwds)

        def clear_cache():
            """Clear the cache."""
            core.clear_cache()

        def clear_being_calculated():
            """Marks all entries in this cache as not being calculated."""
            core.clear_being_calculated()

        func_wrapper.clear_cache = clear_cache
        func_wrapper.clear_being_calculated = clear_being_calculated
        return func_wrapper

    return _cachier_decorator
