"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.RunInstanceSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const number_variable_1 = require("../../interface/variables/number-variable");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const wait_for_and_assert_1 = require("../../parent-steps/automation/sub-steps/wait-for-and-assert");
const prune_and_transform_record_1 = require("../../utils/prune-and-transform-record");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
/**
 * AutomationStep implemenation for aws:runInstance
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-runinstance.html
 */
class RunInstanceSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:runInstances';
        this.runInstanceStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        console.log(`RunInstance: Creating instances of ${this.runInstanceStep.imageId}`);
        const instanceIds = this.runInstances(inputs);
        console.log(`RunInstance: ${this.runInstanceStep.imageId} instances successfully created`);
        this.waitForInstanceRunning(instanceIds);
        console.log(`RunInstance: ${this.runInstanceStep.imageId} instances successfully running`);
        return {
            InstanceIds: instanceIds,
            InstanceStates: ['running'],
        };
    }
    runInstances(inputs) {
        const apiParamMap = {
            AdditionalInfo: this.runInstanceStep.additionalInfo,
            BlockDeviceMapping: this.runInstanceStep.blockDeviceMappings,
            ClientToken: this.runInstanceStep.clientToken,
            DisableApiTermination: this.runInstanceStep.disableApiTermination,
            EbsOptimized: this.runInstanceStep.ebsOptimized,
            IamInstanceProfile: this.runInstanceStep.iamInstanceProfileArn ?? this.runInstanceStep.iamInstanceProfileName,
            ImageId: this.runInstanceStep.imageId,
            InstanceInitiatedShutdownBehavior: this.runInstanceStep.instanceInitiatedShutdownBehavior,
            InstanceType: this.runInstanceStep.instanceType,
            KernalId: this.runInstanceStep.kernelId,
            KeyName: this.runInstanceStep.keyName,
            MaxCount: this.runInstanceStep.maxInstanceCount ?? RunInstanceSimulation.DefaultNumberOfInstances,
            MinCount: this.runInstanceStep.minInstanceCount ?? RunInstanceSimulation.DefaultNumberOfInstances,
            Monitoring: this.runInstanceStep.monitoring,
            NetworkInterfaces: this.runInstanceStep.networkInterfaces,
            Placement: this.runInstanceStep.placement,
            PrivateIpAddress: this.runInstanceStep.privateIpAddress,
            RamdiskId: this.runInstanceStep.ramdiskId,
            SecurityGroups: this.runInstanceStep.securityGroups,
            SecurityGroupIds: this.runInstanceStep.securityGroupIds,
            SubnetId: this.runInstanceStep.subnetId,
            TagSpecifications: this.runInstanceStep.tagSpecifications,
            UserData: this.runInstanceStep.userData,
        };
        const apiParams = prune_and_transform_record_1.pruneAndTransformRecord(apiParamMap, x => x.resolve(inputs));
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'runInstances', {
            service: 'EC2',
            pascalCaseApi: 'RunInstances',
            apiParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    name: 'Instances',
                    selector: '$.Instances',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Run instances failed for ${this.runInstanceStep.imageId}`);
        }
        return result.outputs?.['runInstances.Instances'].map((instance) => instance.InstanceId) ?? [];
    }
    waitForInstanceRunning(instanceIds) {
        wait_for_and_assert_1.waitForAndAssertInstanceState({
            waitForState: ['running', 'shutting-down', 'terminated', 'stopping', 'stopped'],
            assertState: 'running',
            instanceIds: instanceIds,
        }, this.props);
        wait_for_and_assert_1.waitForAndAssertInstanceStatus({
            waitForStatus: ['ok', 'impaired', 'insufficient-data', 'not-applicable'],
            assertStatus: 'ok',
            instanceIds: instanceIds,
        }, this.props);
        const apiParams = {
            InstanceIds: instanceIds,
            IncludeAllInstances: true,
        };
        const stepParams = {
            service: 'EC2',
            pascalCaseApi: 'DescribeInstanceStatus',
            apiParams,
        };
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'getInstanceStates', {
            ...stepParams,
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.MAP_LIST,
                    name: 'InstanceStatuses',
                    selector: '$.InstanceStatuses',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Get instance status failed for ${this.runInstanceStep.imageId} instances`);
        }
        return result.outputs?.['getInstanceStates.InstanceStatuses'].map((instance) => instance.InstanceState.Name) ?? [];
    }
}
exports.RunInstanceSimulation = RunInstanceSimulation;
_a = JSII_RTTI_SYMBOL_1;
RunInstanceSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.RunInstanceSimulation", version: "0.0.20" };
RunInstanceSimulation.DefaultNumberOfInstances = new number_variable_1.HardCodedNumber(1);
//# sourceMappingURL=data:application/json;base64,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