"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringStep = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const choice_1 = require("../../domain/choice");
const data_type_1 = require("../../domain/data-type");
const operation_1 = require("../../domain/operation");
const string_variable_1 = require("../../interface/variables/string-variable");
const assert_aws_resource_step_1 = require("../../parent-steps/automation/assert-aws-resource-step");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const branch_step_1 = require("../../parent-steps/automation/branch-step");
const execute_script_step_1 = require("../../parent-steps/automation/execute-script-step");
const pause_step_1 = require("../../parent-steps/automation/pause-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const wait_for_resource_step_1 = require("../../parent-steps/automation/wait-for-resource-step");
const composite_step_1 = require("./composite-step");
const approve_step_1 = require("../../parent-steps/automation/approve-step");
// eslint-disable-next-line
const yaml = require('js-yaml');
/**
 * StringStep allows for including a step from an existing automation document verbatim.
 * This is useful in that it allows developers to integrate with existing document steps.
 * This step can be used just as you would use any other Step including simulation and deployment.
 */
class StringStep extends composite_step_1.CompositeAutomationStep {
    constructor(scope, id, props) {
        super(scope, id);
        const sharedProps = {};
        sharedProps.name = props.name;
        sharedProps.action = props.action;
        if (props.maxAttempts) {
            sharedProps.maxAttempts = props.maxAttempts;
        }
        if (props.timeoutSeconds) {
            sharedProps.timeoutSeconds = props.timeoutSeconds;
        }
        if (props.isEnd != undefined) {
            sharedProps.isEnd = props.isEnd;
        }
        if (props.isCritical != undefined) {
            sharedProps.isCritical = props.isCritical;
        }
        if (props.outputs != undefined) {
            sharedProps.outputs = this.toOutputs(props.outputs);
        }
        const { Service, Api, PropertySelector, DesiredValues, ...restParams } = props.inputs;
        switch (props.action) {
            case 'aws:executeAwsApi':
                this.automationStep = new aws_api_step_1.AwsApiStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    outputs: sharedProps.outputs,
                    ...sharedProps,
                });
                break;
            case 'aws:waitForAwsResourceProperty':
                this.automationStep = new wait_for_resource_step_1.WaitForResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:assertAwsResourceProperty':
                this.automationStep = new assert_aws_resource_step_1.AssertAwsResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:pause':
                this.automationStep = new pause_step_1.PauseStep(this, props.name, {
                    ...sharedProps,
                });
                break;
            case 'aws:sleep':
                this.automationStep = new sleep_step_1.SleepStep(this, props.name, {
                    sleepSeconds: this.parseDurationToSeconds(restParams.Duration),
                    ...sharedProps,
                });
                break;
            case 'aws:executeScript':
                this.automationStep = new execute_script_step_1.ExecuteScriptStep(this, props.name, {
                    language: execute_script_step_1.ExecuteScriptStep.getLanguage(restParams.Runtime),
                    inputs: Object.keys(restParams.InputPayload),
                    inlineCode: restParams.Script,
                    handlerName: restParams.Handler,
                    ...sharedProps,
                });
                break;
            case 'aws:branch':
                this.automationStep = new branch_step_1.BranchStep(this, props.name, {
                    choices: this.toChoices(restParams.Choices),
                    defaultStepName: restParams.Default,
                    ...sharedProps,
                });
                break;
            case 'aws:approve':
                this.automationStep = new approve_step_1.ApproveStep(this, props.name, {
                    approvers: restParams.Approvers,
                    minRequiredApprovals: restParams.MinRequiredApprovals,
                    message: restParams.Message,
                    notificationArn: restParams.NotificationArn,
                    ...sharedProps,
                });
                break;
            default:
                throw new Error('Action not implemented as StringStep ' + props.action);
        }
    }
    /**
       * Builds a step object from a yaml declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromYaml(scope, inputYaml) {
        return StringStep.fromObject(scope, yaml.load(inputYaml));
    }
    /**
       * Builds a step object from a json declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromJson(scope, json) {
        return StringStep.fromObject(scope, JSON.parse(json));
    }
    /**
       * Builds a step object from an object.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromObject(scope, props) {
        return new StringStep(scope, props.name, props);
    }
    addToDocument(doc) {
        doc.addStep(this.automationStep);
    }
    toOutputs(declaredOutputs) {
        return declaredOutputs.map(declaredOutput => this.toOutput(declaredOutput));
    }
    toOutput(declaredOutput) {
        return {
            name: declaredOutput.Name,
            selector: declaredOutput.Selector,
            outputType: data_type_1.DataType.fromDataType(declaredOutput.Type).dataTypeEnum,
        };
    }
    toChoices(declaredChoices) {
        return declaredChoices.map(declaredChoice => this.toChoice(declaredChoice));
    }
    toChoice(declaredChoice) {
        const operationEntry = Object.entries(declaredChoice)
            .filter(entry => operation_1.OperationEvaluator.STRING_TO_OPERATION[entry[0]] != undefined)[0];
        return new choice_1.Choice({
            jumpToStepName: declaredChoice.NextStep,
            variable: this.toVariable(declaredChoice.Variable),
            operation: operation_1.OperationEvaluator.fromOperationName(operationEntry[0]),
            constant: operationEntry[1],
        });
    }
    toVariable(variable) {
        if (variable.includes('{{')) {
            return new string_variable_1.StringVariable(variable.replace('{{', '').replace('}}', '').trim());
        }
        return new string_variable_1.HardCodedString(variable.trim());
    }
    parseDurationToSeconds(iso8601Duration) {
        // https://stackoverflow.com/a/29153059
        const iso8601DurationRegex = /(-)?P(?:([.,\d]+)Y)?(?:([.,\d]+)M)?(?:([.,\d]+)W)?(?:([.,\d]+)D)?T(?:([.,\d]+)H)?(?:([.,\d]+)M)?(?:([.,\d]+)S)?/;
        const matches = iso8601Duration.match(iso8601DurationRegex);
        if (matches == undefined) {
            throw new Error('Could not parse Duration' + iso8601Duration);
        }
        const periods = {
            years: matches[2] === undefined ? 0 : Number.parseInt(matches[2]),
            months: matches[3] === undefined ? 0 : Number.parseInt(matches[3]),
            weeks: matches[4] === undefined ? 0 : Number.parseInt(matches[4]),
            days: matches[5] === undefined ? 0 : Number.parseInt(matches[5]),
            hours: matches[6] === undefined ? 0 : Number.parseInt(matches[6]),
            minutes: matches[7] === undefined ? 0 : Number.parseInt(matches[7]),
            seconds: matches[8] === undefined ? 0 : Number.parseInt(matches[8]),
        };
        return periods.seconds +
            periods.minutes * 60 +
            periods.hours * 60 * 60 +
            periods.days * 60 * 60 * 24 +
            periods.weeks * 60 * 60 * 24 * 7 +
            periods.months * 60 * 60 * 24 * 30.5 +
            periods.years * 60 * 60 * 24 * 365;
    }
    ;
}
exports.StringStep = StringStep;
_a = JSII_RTTI_SYMBOL_1;
StringStep[_a] = { fqn: "@cdklabs/cdk-ssm-documents.StringStep", version: "0.0.20" };
//# sourceMappingURL=data:application/json;base64,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