"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SPADeploy = void 0;
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config != 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config, isForCloudFront) {
        let bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true || isForCloudFront === true) {
            bucketConfig.publicReadAccess = false;
        }
        let bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true && isForCloudFront === false) {
            if (typeof this.globalConfig.ipList == 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(bucket.bucketArn + '/*');
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, accessIdentity, cert) {
        let cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket,
                        originAccessIdentity: accessIdentity
                    },
                    behaviors: [{ isDefaultBehavior: true }]
                }
            ],
            //We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? '/' + config.errorDoc : '/' + config.indexDoc),
                    responseCode: 200
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? '/' + config.errorDoc : '/' + config.indexDoc),
                    responseCode: 200
                }]
        };
        if (typeof config.certificateARN != 'undefined' && typeof config.cfAliases != 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases
            };
        }
        if (typeof config.zoneName != 'undefined' && typeof cert != 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config, false);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        let cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName == 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution
     * It will also setup error forwarding and unauth forwarding back to indexDoc
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            //Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution: distribution,
            distributionPaths: ['/', '/' + config.indexDoc]
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.domainName
        });
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto
     * configured by using the details in the hosted zone provided
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        let zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        let cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName: config.zoneName,
            region: 'us-east-1',
        });
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            //Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution: distribution,
            distributionPaths: ['/', '/' + config.indexDoc]
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: config.zoneName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution))
        });
        new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
            zone,
            recordNames: ['www.' + config.zoneName],
            targetDomain: config.zoneName,
        });
    }
}
exports.SPADeploy = SPADeploy;
//# sourceMappingURL=data:application/json;base64,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