/*
 * Decompiled with CFR 0.152.
 */
package org.mapsforge.map.writer.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiLineString;
import org.locationtech.jts.geom.MultiPolygon;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;
import org.locationtech.jts.geom.TopologyException;
import org.locationtech.jts.simplify.TopologyPreservingSimplifier;
import org.mapsforge.core.model.BoundingBox;
import org.mapsforge.core.model.LatLong;
import org.mapsforge.core.util.LatLongUtils;
import org.mapsforge.core.util.MercatorProjection;
import org.mapsforge.map.writer.model.TDNode;
import org.mapsforge.map.writer.model.TDWay;
import org.mapsforge.map.writer.model.TileCoordinate;
import org.mapsforge.map.writer.model.WayDataBlock;
import org.mapsforge.map.writer.util.JTSUtils;

public final class GeoUtils {
    public static final int MIN_COORDINATES_POLYGON = 8;
    public static final int MIN_NODES_POLYGON = 4;
    private static final double[] EPSILON_ZERO = new double[]{0.0, 0.0};
    private static final GeometryFactory GEOMETRY_FACTORY = new GeometryFactory();
    private static final Logger LOGGER = Logger.getLogger(GeoUtils.class.getName());
    private static final byte SUBTILE_ZOOMLEVEL_DIFFERENCE = 2;
    private static final int[] TILE_BITMASK_VALUES = new int[]{32768, 16384, 8192, 4096, 2048, 1024, 512, 256, 128, 64, 32, 16, 8, 4, 2, 1};

    public static Geometry clipToTile(TDWay way, Geometry geometry, TileCoordinate tileCoordinate, int enlargementInMeters) {
        Geometry tileBBJTS = null;
        Geometry ret = null;
        tileBBJTS = GeoUtils.tileToJTSGeometry(tileCoordinate.getX(), tileCoordinate.getY(), tileCoordinate.getZoomlevel(), enlargementInMeters);
        try {
            if (!geometry.isValid()) {
                LOGGER.warning("invalid geometry prior to tile clipping, trying to repair " + way.getId());
                geometry = JTSUtils.repairInvalidPolygon(geometry);
                if (!geometry.isValid()) {
                    LOGGER.warning("invalid geometry even after attempt to fix " + way.getId());
                }
            }
            if (((ret = tileBBJTS.intersection(geometry)) instanceof Polygon || ret instanceof MultiPolygon) && !ret.isValid()) {
                LOGGER.warning("clipped way is not valid, trying to repair it: " + way.getId());
                ret = JTSUtils.repairInvalidPolygon(ret);
                if (ret == null) {
                    way.setInvalid(true);
                    LOGGER.warning("could not repair invalid polygon: " + way.getId());
                }
            }
        }
        catch (TopologyException e) {
            LOGGER.log(Level.WARNING, "JTS cannot clip way, not storing it in data file: " + way.getId(), e);
            way.setInvalid(true);
            return null;
        }
        return ret;
    }

    public static short computeBitmask(Geometry geometry, TileCoordinate tile, int enlargementInMeter) {
        List<TileCoordinate> subtiles = tile.translateToZoomLevel((byte)(tile.getZoomlevel() + 2));
        short bitmask = 0;
        int tileCounter = 0;
        for (TileCoordinate subtile : subtiles) {
            Geometry bbox = GeoUtils.tileToJTSGeometry(subtile.getX(), subtile.getY(), subtile.getZoomlevel(), enlargementInMeter);
            if (bbox.intersects(geometry)) {
                bitmask = (short)(bitmask | TILE_BITMASK_VALUES[tileCounter]);
            }
            ++tileCounter;
        }
        return bitmask;
    }

    public static LatLong computeCentroid(Geometry geometry) {
        Point centroid = geometry.getCentroid();
        if (centroid != null) {
            return new LatLong(centroid.getCoordinate().y, centroid.getCoordinate().x);
        }
        return null;
    }

    public static LatLong computeInteriorPoint(Geometry geometry) {
        Point interiorPoint = geometry.getInteriorPoint();
        if (interiorPoint != null) {
            return new LatLong(interiorPoint.getCoordinate().y, interiorPoint.getCoordinate().x);
        }
        return null;
    }

    public static boolean coveredByTile(Geometry geometry, TileCoordinate tile, int enlargementInMeter) {
        Geometry bbox = GeoUtils.tileToJTSGeometry(tile.getX(), tile.getY(), tile.getZoomlevel(), enlargementInMeter);
        return bbox.covers(geometry);
    }

    public static BoundingBox mapWayToBoundingBox(TDWay way) {
        TDNode[] wayNodes = way.getWayNodes();
        if (wayNodes.length < 3) {
            return null;
        }
        try {
            double lat = LatLongUtils.microdegreesToDegrees(wayNodes[0].getLatitude());
            double lon = LatLongUtils.microdegreesToDegrees(wayNodes[0].getLongitude());
            BoundingBox bb = new BoundingBox(lat, lon, lat, lon);
            for (int i = 1; i < wayNodes.length; ++i) {
                bb = bb.extendCoordinates(LatLongUtils.microdegreesToDegrees(wayNodes[i].getLatitude()), LatLongUtils.microdegreesToDegrees(wayNodes[i].getLongitude()));
            }
            return bb;
        }
        catch (IllegalArgumentException ex) {
            LOGGER.warning("wrong coordinates on way: " + way.toString() + "\nLat: " + LatLongUtils.microdegreesToDegrees(wayNodes[0].getLatitude()) + " Lon: " + LatLongUtils.microdegreesToDegrees(wayNodes[0].getLongitude()));
            return null;
        }
    }

    public static Polygon mapWayToPolygon(TDWay way) {
        TDNode[] wayNodes = way.getWayNodes();
        if (wayNodes.length < 3) {
            return null;
        }
        Coordinate[] wayCoords = new Coordinate[wayNodes.length + 1];
        for (int i = 0; i < wayCoords.length - 1; ++i) {
            wayCoords[i] = new Coordinate(wayNodes[i].getLatitude(), wayNodes[i].getLongitude());
        }
        wayCoords[wayCoords.length - 1] = wayCoords[0];
        Polygon polygon = GEOMETRY_FACTORY.createPolygon(GEOMETRY_FACTORY.createLinearRing(wayCoords), null);
        if (!polygon.isValid()) {
            return null;
        }
        return polygon;
    }

    public static Set<TileCoordinate> mapWayToTiles(TDWay way, byte baseZoomLevel, int enlargementInMeter) {
        if (way == null) {
            LOGGER.fine("way is null in mapping to tiles");
            return Collections.emptySet();
        }
        HashSet<TileCoordinate> matchedTiles = new HashSet<TileCoordinate>();
        Geometry wayGeometry = JTSUtils.toJTSGeometry(way);
        if (wayGeometry == null) {
            way.setInvalid(true);
            LOGGER.fine("unable to create geometry from way: " + way.getId());
            return matchedTiles;
        }
        TileCoordinate[] bbox = GeoUtils.getWayBoundingBox(way, baseZoomLevel, enlargementInMeter);
        try {
            for (int k = bbox[0].getX(); k <= bbox[1].getX(); ++k) {
                for (int l = bbox[0].getY(); l <= bbox[1].getY(); ++l) {
                    Geometry bboxGeometry = GeoUtils.tileToJTSGeometry(k, l, baseZoomLevel, enlargementInMeter);
                    if (!bboxGeometry.intersects(wayGeometry)) continue;
                    matchedTiles.add(new TileCoordinate(k, l, baseZoomLevel));
                }
            }
        }
        catch (TopologyException e) {
            LOGGER.fine("encountered error during mapping of a way to corresponding tiles, way id: " + way.getId());
            return Collections.emptySet();
        }
        return matchedTiles;
    }

    public static boolean pointInTile(LatLong latLong, TileCoordinate tile) {
        if (latLong == null || tile == null) {
            return false;
        }
        double lon1 = MercatorProjection.tileXToLongitude(tile.getX(), tile.getZoomlevel());
        double lon2 = MercatorProjection.tileXToLongitude(tile.getX() + 1, tile.getZoomlevel());
        double lat1 = MercatorProjection.tileYToLatitude(tile.getY(), tile.getZoomlevel());
        double lat2 = MercatorProjection.tileYToLatitude(tile.getY() + 1, tile.getZoomlevel());
        return latLong.latitude <= lat1 && latLong.latitude >= lat2 && latLong.longitude >= lon1 && latLong.longitude <= lon2;
    }

    public static Geometry simplifyGeometry(TDWay way, Geometry geometry, byte zoomlevel, int tileSize, double simplificationFactor) {
        Geometry ret = null;
        Envelope bbox = geometry.getEnvelopeInternal();
        double latMax = Math.max(Math.abs(bbox.getMaxY()), Math.abs(bbox.getMinY()));
        double deltaLat = GeoUtils.deltaLat(simplificationFactor, latMax, zoomlevel, tileSize);
        try {
            ret = TopologyPreservingSimplifier.simplify(geometry, deltaLat);
        }
        catch (TopologyException e) {
            LOGGER.log(Level.FINE, "JTS cannot simplify way due to an error, not simplifying way with id: " + way.getId(), e);
            way.setInvalid(true);
            return geometry;
        }
        return ret;
    }

    public static List<WayDataBlock> toWayDataBlockList(Geometry geometry) {
        ArrayList<WayDataBlock> res;
        block8: {
            block11: {
                block10: {
                    block9: {
                        block7: {
                            res = new ArrayList<WayDataBlock>();
                            if (!(geometry instanceof MultiPolygon)) break block7;
                            MultiPolygon mp = (MultiPolygon)geometry;
                            for (int i = 0; i < mp.getNumGeometries(); ++i) {
                                Polygon p = (Polygon)mp.getGeometryN(i);
                                List<Integer> outer = GeoUtils.toCoordinateList(p.getExteriorRing());
                                if (outer.size() / 2 <= 0) continue;
                                ArrayList<List<Integer>> inner = new ArrayList<List<Integer>>();
                                for (int j = 0; j < p.getNumInteriorRing(); ++j) {
                                    List<Integer> innr = GeoUtils.toCoordinateList(p.getInteriorRingN(j));
                                    if (innr.size() / 2 <= 0) continue;
                                    inner.add(innr);
                                }
                                res.add(new WayDataBlock(outer, inner));
                            }
                            break block8;
                        }
                        if (!(geometry instanceof Polygon)) break block9;
                        Polygon p = (Polygon)geometry;
                        List<Integer> outer = GeoUtils.toCoordinateList(p.getExteriorRing());
                        if (outer.size() / 2 <= 0) break block8;
                        ArrayList<List<Integer>> inner = new ArrayList<List<Integer>>();
                        for (int i = 0; i < p.getNumInteriorRing(); ++i) {
                            List<Integer> innr = GeoUtils.toCoordinateList(p.getInteriorRingN(i));
                            if (innr.size() / 2 <= 0) continue;
                            inner.add(innr);
                        }
                        res.add(new WayDataBlock(outer, inner));
                        break block8;
                    }
                    if (!(geometry instanceof MultiLineString)) break block10;
                    MultiLineString ml = (MultiLineString)geometry;
                    for (int i = 0; i < ml.getNumGeometries(); ++i) {
                        LineString l = (LineString)ml.getGeometryN(i);
                        List<Integer> outer = GeoUtils.toCoordinateList(l);
                        if (outer.size() / 2 <= 0) continue;
                        res.add(new WayDataBlock(outer, null));
                    }
                    break block8;
                }
                if (!(geometry instanceof LinearRing) && !(geometry instanceof LineString)) break block11;
                List<Integer> outer = GeoUtils.toCoordinateList(geometry);
                if (outer.size() / 2 <= 0) break block8;
                res.add(new WayDataBlock(outer, null));
                break block8;
            }
            if (geometry instanceof GeometryCollection) {
                GeometryCollection gc = (GeometryCollection)geometry;
                for (int i = 0; i < gc.getNumGeometries(); ++i) {
                    List<WayDataBlock> recursiveResult = GeoUtils.toWayDataBlockList(gc.getGeometryN(i));
                    for (WayDataBlock wayDataBlock : recursiveResult) {
                        List<Integer> outer = wayDataBlock.getOuterWay();
                        if (outer.size() / 2 <= 0) continue;
                        res.add(wayDataBlock);
                    }
                }
            }
        }
        return res;
    }

    private static double[] bufferInDegrees(long tileY, byte zoom, int enlargementInMeter) {
        if (enlargementInMeter == 0) {
            return EPSILON_ZERO;
        }
        double[] epsilons = new double[2];
        double lat = MercatorProjection.tileYToLatitude(tileY, zoom);
        epsilons[0] = LatLongUtils.latitudeDistance(enlargementInMeter);
        epsilons[1] = LatLongUtils.longitudeDistance(enlargementInMeter, lat);
        return epsilons;
    }

    private static double[] computeTileEnlargement(double lat, int enlargementInMeter) {
        if (enlargementInMeter == 0) {
            return EPSILON_ZERO;
        }
        double[] epsilons = new double[]{LatLongUtils.latitudeDistance(enlargementInMeter), LatLongUtils.longitudeDistance(enlargementInMeter, lat)};
        return epsilons;
    }

    private static double deltaLat(double deltaPixel, double lat, byte zoom, int tileSize) {
        long mapSize = MercatorProjection.getMapSize(zoom, tileSize);
        double pixelY = MercatorProjection.latitudeToPixelY(lat, mapSize);
        double lat2 = MercatorProjection.pixelYToLatitude(pixelY + deltaPixel, mapSize);
        return Math.abs(lat2 - lat);
    }

    private static TileCoordinate[] getWayBoundingBox(TDWay way, byte zoomlevel, int enlargementInMeter) {
        double maxx = Double.NEGATIVE_INFINITY;
        double maxy = Double.NEGATIVE_INFINITY;
        double minx = Double.POSITIVE_INFINITY;
        double miny = Double.POSITIVE_INFINITY;
        for (TDNode coordinate : way.getWayNodes()) {
            maxy = Math.max(maxy, LatLongUtils.microdegreesToDegrees(coordinate.getLatitude()));
            miny = Math.min(miny, LatLongUtils.microdegreesToDegrees(coordinate.getLatitude()));
            maxx = Math.max(maxx, LatLongUtils.microdegreesToDegrees(coordinate.getLongitude()));
            minx = Math.min(minx, LatLongUtils.microdegreesToDegrees(coordinate.getLongitude()));
        }
        double[] epsilonsTopLeft = GeoUtils.computeTileEnlargement(maxy, enlargementInMeter);
        double[] epsilonsBottomRight = GeoUtils.computeTileEnlargement(miny, enlargementInMeter);
        TileCoordinate[] bbox = new TileCoordinate[]{new TileCoordinate(MercatorProjection.longitudeToTileX(minx - epsilonsTopLeft[1], zoomlevel), MercatorProjection.latitudeToTileY(maxy + epsilonsTopLeft[0], zoomlevel), zoomlevel), new TileCoordinate(MercatorProjection.longitudeToTileX(maxx + epsilonsBottomRight[1], zoomlevel), MercatorProjection.latitudeToTileY(miny - epsilonsBottomRight[0], zoomlevel), zoomlevel)};
        return bbox;
    }

    private static Geometry tileToJTSGeometry(long tileX, long tileY, byte zoom, int enlargementInMeter) {
        double minLat = MercatorProjection.tileYToLatitude(tileY + 1L, zoom);
        double maxLat = MercatorProjection.tileYToLatitude(tileY, zoom);
        double minLon = MercatorProjection.tileXToLongitude(tileX, zoom);
        double maxLon = MercatorProjection.tileXToLongitude(tileX + 1L, zoom);
        double[] epsilons = GeoUtils.bufferInDegrees(tileY, zoom, enlargementInMeter);
        Coordinate bottomLeft = new Coordinate(minLon -= epsilons[1], minLat -= epsilons[0]);
        Coordinate topRight = new Coordinate(maxLon += epsilons[1], maxLat += epsilons[0]);
        return GEOMETRY_FACTORY.createLineString(new Coordinate[]{bottomLeft, topRight}).getEnvelope();
    }

    private static List<Integer> toCoordinateList(Geometry jtsGeometry) {
        Coordinate[] jtsCoords = jtsGeometry.getCoordinates();
        ArrayList<Integer> result = new ArrayList<Integer>();
        for (int j = 0; j < jtsCoords.length; ++j) {
            LatLong latLong = new LatLong(jtsCoords[j].y, jtsCoords[j].x);
            result.add(LatLongUtils.degreesToMicrodegrees(latLong.latitude));
            result.add(LatLongUtils.degreesToMicrodegrees(latLong.longitude));
        }
        return result;
    }

    private GeoUtils() {
    }
}

