"""
! Name: yolov5_onnx.py
! Author: lolisky
! Date: 2022-04-28

功能：为智能分析提供物体识别支持（基于Yolov5）
"""
import cv2
import numpy as np
import time
import os
from numpy import array


class Colors:       # 颜色类
    def __init__(self):
        colors = ('FF3838', '344593', '0018EC', 'FFB21D', 'CFD231', 'CB38FF', '92CC17', '2C99A8', '1A9334', '00D4BB',
               '3DDB86', '00C2FF', 'FF9D97', '6473FF', 'FF701F', '8438FF', '520085', '48F90A', 'FF95C8', 'FF37C7')
        self.palette = [self.hex2rgb('#' + c) for c in colors]
        self.n = len(self.palette)

    def __call__(self, i, bgr=False):
        c = self.palette[int(i) % self.n]
        return (c[2], c[1], c[0]) if bgr else c

    @staticmethod
    def hex2rgb(h):         # 色彩空间转换
        return tuple(int(h[1 + i:1 + i + 2], 16) for i in (0, 2, 4))

colors = Colors()


class yolov5():     # 核心物体识别类
    def __init__(self, onnx_path, confThreshold=0.34, nmsThreshold=0.45):
        self.classes = [
        'person', 'bicycle', 'car', 'motorcycle', 'airplane', 'bus', 'train', 'truck', 'boat', 'traffic light',
        'fire hydrant', 'stop sign', 'parking meter', 'bench', 'bird', 'cat', 'dog', 'horse', 'sheep', 'cow',
        'elephant', 'bear', 'zebra', 'giraffe', 'backpack', 'umbrella', 'handbag', 'tie', 'suitcase', 'frisbee',
        'skis', 'snowboard', 'sports ball', 'kite', 'baseball bat', 'baseball glove', 'skateboard', 'surfboard',
        'tennis racket', 'bottle', 'wine glass', 'cup', 'fork', 'knife', 'spoon', 'bowl', 'banana', 'apple',
        'sandwich', 'orange', 'broccoli', 'carrot', 'hot dog', 'pizza', 'donut', 'cake', 'chair', 'couch',
        'potted plant', 'bed', 'dining table', 'toilet', 'tv', 'laptop', 'mouse', 'remote', 'keyboard', 'cell phone',
        'microwave', 'oven', 'toaster', 'sink', 'refrigerator', 'book', 'clock', 'vase', 'scissors', 'teddy bear',
        'hair drier', 'toothbrush'
        ]       # 物体分类
        self.stride = np.array([8., 16., 32.])
        self.inpWidth = 640
        self.inpHeight = 640
        self.colors = [np.random.randint(0, 255, size=3).tolist() for _ in range(len(self.classes))]
        num_classes = len(self.classes)
        self.anchors = [[10, 13, 16, 30, 33, 23], [30, 61, 62, 45, 59, 119], [116, 90, 156, 198, 373, 326]]
        self.nl = len(self.anchors)
        self.na = len(self.anchors[0]) // 2
        self.no = num_classes + 5
        self.confThreshold = confThreshold
        self.nmsThreshold = nmsThreshold
        self.net = cv2.dnn.readNetFromONNX(onnx_path)

        self.confThreshold = confThreshold
        self.nmsThreshold = nmsThreshold
    
    def _make_grid(self, nx=20, ny=20):
        xv, yv = np.meshgrid(np.arange(ny), np.arange(nx))
        return np.stack((xv, yv), 2).reshape((-1, 2)).astype(np.float32)

    def letterbox(self, im, new_shape=(640, 640), color=(114, 114, 114), auto=True, scaleFill=False, scaleup=True, stride=32):
     # 重构图片大小
        shape = im.shape[:2]  # 当前的大小
        if isinstance(new_shape, int):
            new_shape = (new_shape, new_shape)

        # 计算放缩比例
        r = min(new_shape[0] / shape[0], new_shape[1] / shape[1])
        if not scaleup:
            r = min(r, 1.0)

        # 完成paddding
        ratio = r, r  # 宽，高，比例
        new_unpad = int(round(shape[1] * r)), int(round(shape[0] * r))
        dw, dh = new_shape[1] - new_unpad[0], new_shape[0] - new_unpad[1]
        if auto:  # 最小方框
            dw, dh = np.mod(dw, stride), np.mod(dh, stride)
        elif scaleFill:
            dw, dh = 0.0, 0.0
            new_unpad = (new_shape[1], new_shape[0])
            ratio = new_shape[1] / shape[1], new_shape[0] / shape[0]

        dw /= 2  # 分为两部分
        dh /= 2

        if shape[::-1] != new_unpad:  # 重构
            im = cv2.resize(im, new_unpad, interpolation=cv2.INTER_LINEAR)
        top, bottom = int(round(dh - 0.1)), int(round(dh + 0.1))
        left, right = int(round(dw - 0.1)), int(round(dw + 0.1))
        im = cv2.copyMakeBorder(im, top, bottom, left, right, cv2.BORDER_CONSTANT, value=color)  # 增加边界
        return im, ratio, (dw, dh)


    def box_area(self,boxes :array):
        return (boxes[:, 2] - boxes[:, 0]) * (boxes[:, 3] - boxes[:, 1])

    def box_iou(self,box1 :array, box2: array):
        area1 = self.box_area(box1)
        area2 = self.box_area(box2)
        # broadcasting, 两个数组各维度大小 从后往前对比一致， 或者 有一维度值为1；
        lt = np.maximum(box1[:, np.newaxis, :2], box2[:, :2])
        rb = np.minimum(box1[:, np.newaxis, 2:], box2[:, 2:])
        wh = rb - lt
        wh = np.maximum(0, wh) # [N, M, 2]
        inter = wh[:, :, 0] * wh[:, :, 1]
        iou = inter / (area1[:, np.newaxis] + area2 - inter)
        return iou  # NxM

    def numpy2nms(self, boxes :array, scores :array, iou_threshold :float):
        idxs = scores.argsort()  # 按分数 降序排列的索引 [N]
        keep = []
        while idxs.size > 0:  # 统计数组中元素的个数
            max_score_index = idxs[-1]
            max_score_box = boxes[max_score_index][None, :]
            keep.append(max_score_index)

            if idxs.size == 1:
                break
            idxs = idxs[:-1]  # 将得分最大框 从索引中删除； 剩余索引对应的框 和 得分最大框 计算IoU；
            other_boxes = boxes[idxs]  # [?, 4]
            ious = self.box_iou(max_score_box, other_boxes)  # 一个框和其余框比较 1XM
            idxs = idxs[ious[0] <= iou_threshold]

        keep = np.array(keep)  
        return keep

    def xywh2xyxy(self,x):
        # 转换 nx4 boxes 从 [x, y, w, h] 变为 [x1, y1, x2, y2] 其中，xy1=top-left, xy2=bottom-right
        y = np.copy(x)
        y[:, 0] = x[:, 0] - x[:, 2] / 2  # top left x
        y[:, 1] = x[:, 1] - x[:, 3] / 2  # top left y
        y[:, 2] = x[:, 0] + x[:, 2] / 2  # bottom right x
        y[:, 3] = x[:, 1] + x[:, 3] / 2  # bottom right y
        return y

    def non_max_suppression(self,prediction, conf_thres=0.34, agnostic=False):                                                 #25200 = 20*20*3 + 40*40*3 + 80*80*3
        xc = prediction[..., 4] > conf_thres  # candidates,获取置信度，prediction为所有的预测结果.shape(1, 25200, 21),batch为1，25200个预测结果，21 = x,y,w,h,c + class个数
        min_wh, max_wh = 2, 4096
        max_nms = 30000
        output = [np.zeros((0, 6))] * prediction.shape[0]
        for xi, x in enumerate(prediction):
            # Apply constraints
            x = x[xc[xi]]  # confidence，获取confidence大于conf_thres的结果
            if not x.shape[0]:
                continue
            x[:, 5:] *= x[:, 4:5]  # conf = obj_conf * cls_conf
            # Box (center x, center y, width, height) to (x1, y1, x2, y2)
            box = self.xywh2xyxy(x[:, :4])
            # Detections matrix nx6 (xyxy, conf, cls)
            conf = np.max(x[:, 5:], axis=1)    #获取类别最高的置信度
            j = np.argmax(x[:, 5:],axis=1)     #获取下标
            #转为array：  x = torch.cat((box, conf, j.float()), 1)[conf.view(-1) > conf_thres]
            re = np.array(conf.reshape(-1)> conf_thres)
            #转为维度
            conf =conf.reshape(-1,1)
            j = j.reshape(-1,1)
            #numpy的拼接
            x = np.concatenate((box,conf,j),axis=1)[re]
            # Check shape
            n = x.shape[0]  # number of boxes
            if not n:  # no boxes
                continue
            elif n > max_nms:  # excess boxes
                x = x[x[:, 4].argsort(descending=True)[:max_nms]]  # sort by confidence
            # Batched NMS
            c = x[:, 5:6] * (0 if agnostic else max_wh)  # classes
            boxes, scores = x[:, :4] + c, x[:, 4]  # boxes (offset by class), scores
            i = self.numpy2nms(boxes, scores, self.nmsThreshold)
            output[xi] = x[i]
        return output

    def detect(self, srcimg):
        result = [[], [], [], [], [], [], [], []]     # 存储识别结果
        im = srcimg.copy()
        im, ratio, wh = self.letterbox(srcimg, self.inpWidth, stride=self.stride, auto=False)
        # Sets the input to the network
        blob = cv2.dnn.blobFromImage(im, 1 / 255.0,swapRB=True, crop=False)
        self.net.setInput(blob)
        outs = self.net.forward(self.net.getUnconnectedOutLayersNames())[0]
        #NMS
        pred = self.non_max_suppression(outs, self.confThreshold,agnostic=False)
        #draw box
        for i in pred[0]:
            left = int((i[0] - wh[0])/ratio[0])     # 左上x（横坐标）
            top = int((i[1]-wh[1])/ratio[1])        # 左上y（纵坐标）
            width = int((i[2] - wh[0])/ratio[0])    # 右下x
            height = int((i[3]-wh[1])/ratio[1])     # 右下y
            conf = i[4]
            classId = i[5]
            c = int(classId)
            # 0：人 26：手提包 39：瓶子 67：手机 15：猫 16：狗 25：伞 46[香蕉]-47[苹果]-50[橘子]
            # 对应DItem:0：人 1：手提包 2：瓶子 3：手机 4：猫 5：狗 6：伞 7:水果
            if c == 0:
                result[0].append([int(left), int(top), int(width),int(height)])
            elif c == 26:
                result[1].append([int(left), int(top), int(width),int(height)])
            elif c == 39:
                result[2].append([int(left), int(top), int(width),int(height)])
            elif c == 67:
                result[3].append([int(left), int(top), int(width),int(height)])
            elif c == 15:
                result[4].append([int(left), int(top), int(width),int(height)])
            elif c == 16:
                result[5].append([int(left), int(top), int(width),int(height)])
            elif c == 25:
                result[6].append([int(left), int(top), int(width),int(height)])
            elif c == 46 or c == 47 or c == 49:
                result[7].append([int(left), int(top), int(width),int(height)])
            if c == 0 or c == 26 or c == 39 or c == 67 or c == 15 or c == 16 or c == 25:
                cv2.rectangle(srcimg, (int(left), int(top)), (int(width),int(height)), colors(classId, True), 2, lineType=cv2.LINE_AA)
                label = '%.2f' % conf
                label = '%s:%s' % (self.classes[int(classId)], label)
                # Display the label at the top of the bounding box
                labelSize, baseLine = cv2.getTextSize(label, cv2.FONT_HERSHEY_SIMPLEX, 0.5, 1)
                top = max(top, labelSize[1])
                cv2.putText(srcimg, label, (int(left-20),int(top - 10)), cv2.FONT_HERSHEY_SIMPLEX, 0.5, (0, 255, 255), thickness=1, lineType=cv2.LINE_AA)  # (98,168,28) (0, 255, 255)
            if c == 46 or c == 47 or c == 49:
                cv2.rectangle(srcimg, (int(left), int(top)), (int(width), int(height)), colors(classId, True), 2,
                              lineType=cv2.LINE_AA)
                label = '%.2f' % conf
                label = '%s:%s' % ("fruit", label)

                labelSize, baseLine = cv2.getTextSize(label, cv2.FONT_HERSHEY_SIMPLEX, 0.5, 1)
                top = max(top, labelSize[1])
                cv2.putText(srcimg, label, (int(left - 20), int(top - 10)), cv2.FONT_HERSHEY_SIMPLEX, 0.5,
                            (0, 255, 255), thickness=1, lineType=cv2.LINE_AA)
        # 显示图像
        cv2.imshow('detected' ,srcimg)
        cv2.waitKey(1)
        return result

def mult_test(onnx_path, imgpath):
    model = yolov5(onnx_path)
    srcimg = cv2.imread(imgpath)
    result = model.detect(srcimg)
    return result


