import subprocess
import time
from pathlib import Path
from fujin.config import Config, HostConfig, Webserver, ProcessConfig, InstallationMode
from fujin.commands.deploy import Deploy
from unittest.mock import patch


def test_full_deployment_flow(vps_container, ssh_key_setup, tmp_path):
    # Mock distfile
    dist_dir = tmp_path / "dist"
    dist_dir.mkdir()
    distfile = dist_dir / "app-0.1.0"
    # Create a dummy executable script (Python HTTP server)
    script_content = """#!/usr/bin/env python3
import http.server
import socketserver
import sys

PORT = 8000
Handler = http.server.SimpleHTTPRequestHandler

with socketserver.TCPServer(("", PORT), Handler) as httpd:
    print("serving at port", PORT)
    httpd.serve_forever()
"""
    distfile.write_text(script_content)
    distfile.chmod(0o755)

    # 1. Setup Config pointing to Docker container
    config = Config(
        app_name="integration-test",
        version="0.1.0",
        build_command="echo 'building'",
        distfile=str(distfile),  # Use absolute path
        installation_mode=InstallationMode.BINARY,
        host=HostConfig(
            ip=vps_container["ip"],
            user=vps_container["user"],
            ssh_port=vps_container["port"],
            _key_filename=str(ssh_key_setup),
            domain_name="localhost",
        ),
        webserver=Webserver(upstream="localhost:8000", enabled=True),
        processes={"web": ProcessConfig(command=f"python3 {distfile.name}")},
    )

    try:
        # 2. Run Deploy
        deploy_cmd = Deploy()
        deploy_cmd.config = config
        deploy_cmd()

        # 3. Verify inside the container
        # Check if service is active
        # The service name is generated by Config.get_unit_template_name("web") -> "integration-test.service"
        service_name = "integration-test.service"

        check_cmd = [
            "docker",
            "exec",
            vps_container["name"],
            "systemctl",
            "is-active",
            service_name,
        ]
        result = subprocess.run(check_cmd, capture_output=True, text=True)

        if result.stdout.strip() != "active":
            subprocess.run(
                [
                    "docker",
                    "exec",
                    vps_container["name"],
                    "systemctl",
                    "status",
                    service_name,
                ],
                check=False,
            )
            subprocess.run(
                [
                    "docker",
                    "exec",
                    vps_container["name"],
                    "journalctl",
                    "-u",
                    service_name,
                    "--no-pager",
                ],
                check=False,
            )

        assert result.stdout.strip() == "active"

        # 4. Verify application is responding
        # We curl localhost:8000 inside the container
        curl_cmd = [
            "docker",
            "exec",
            vps_container["name"],
            "curl",
            "-s",
            "-o",
            "/dev/null",
            "-w",
            "%{http_code}",
            "http://localhost:8000",
        ]
        # Retry a few times as startup might take a moment
        for _ in range(10):
            result = subprocess.run(curl_cmd, capture_output=True, text=True)
            if result.stdout.strip() == "200":
                break
            time.sleep(1)

        assert result.stdout.strip() == "200"

    finally:
        # Cleanup local artifact
        if Path("dist/app-0.1.0.whl").exists():
            Path("dist/app-0.1.0.whl").unlink()
        if Path("dist").exists() and not any(Path("dist").iterdir()):
            Path("dist").rmdir()
