# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['boto3_post_conditions', 'boto3_post_conditions.services']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.20,<2.0', 'retry>=0.9.2,<0.10.0']

setup_kwargs = {
    'name': 'boto3-post-conditions',
    'version': '0.2.0',
    'description': 'Remove retry logic from boto3 clients by enforcing post-conditions.',
    'long_description': '# boto3-post-conditions\n\n ![ci](https://github.com/jeking3/boto3_post_conditions/actions/workflows/test.yml/badge.svg)\n[![codecov](https://codecov.io/gh/jeking3/boto3_post_conditions/branch/main/graph/badge.svg?token=NP7WihxzHD)](https://codecov.io/gh/jeking3/boto3_post_conditions)\n[![open issues](https://img.shields.io/github/issues-raw/jeking3/boto3_post_conditions)](https://github.com/jeking3/boto3_post_conditions/issues)\n[![license](https://img.shields.io/badge/License-Apache%202.0-blue.svg)](https://opensource.org/licenses/Apache-2.0)\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n\nMost AWS APIs are eventually consistent with few guarantees.  For example when\nyou use SSM `put_parameter` (which succeeds), then immediately call `get_parameter`,\nyou might get an error such as `ParameterNotFound`.  Welcome to the world of\neventual consistency.  Anyone who has worked with the AWS CLI or API has been\nforced to do one of two things:\n\n1. Add sleep logic to wait for things to align, or\n2. Add retry logic to wait for things to align.\n\nThis library injects logic for post-condition blocking checks in boto3\nto eliminate the majority of retry code that everyone has been forced to\nadd (and develop tests for) in their own code.  This is especially useful\nin serial scripts that manipulate AWS, where different commands in the script\ndepend on changes from the previous command to be fully realized.  Hopefully\nthis sort of logic can make its way into botocore in the future so that\nserialized scripts can take advantage of it and become vastly simplified.\n\n## Quick Start\n\n### Compatibility\n\nboto3-post-conditions supports at least some of the following AWS subsystems:\n\n- SSM Parameter Store\n- Secrets Manager\n\n### Installing\n\n    pip install boto3-post-conditions\n\nor\n\n    poetry add boto3-post-conditions\n\n### Using\n\nCreate a boto3 client like you normally would.  To enforce post-conditions\non calls to that client, register the client with the `PostConditionEnforcer`:\n\n    import boto3\n    from boto3_post_conditions import PostConditionEnforcer\n\n    client = boto3.client("ssm")\n    PostConditionEnforcer.register(client)\n\nThe enforcer will inject event handlers into the client definition to\nblock returning from your API calls until changes are actually realized\nin the AWS service you are modifying.\n\n## Limitations\n\nIt\'s pre-alpha, so there are going to be some major gaps...\n\n- Only supports some of Secrets Manager and SSM initially.\n- Post-conditions are always checked serially after each modification.\n  There is no batch post-condition processing yet, but certainly\n  being able to wrap boto calls in a context manager and then finalize\n  their changes all at once (allowing AWS to process all the requests\n  in parallel) is going to be useful.\n\n## Background\n\nEvery developer who has worked with the AWS API has encountered eventual\nconsistency issues.  This behavior has led virtually every reliable product\nthat leverages the AWS API to add their own code to defeat the effects of\neventual consistency.  The best way to deal with eventual consistency in a\npipeline is to add post-condition checks.  For example, if you delete a secret\nin AWS Secrets Manager, it is not immediately deleted.  It is scheduled for\ndeletion and then it is eventually deleted.  If you attempt to insert a new\nsecret with the same name during this transition you get an error.  You may\neven still be able to read the secret even though the delete call returned\nsuccess.\n\nAWS documents [eventual consistency](https://docs.aws.amazon.com/AWSEC2/latest/APIReference/query-api-troubleshooting.html#eventual-consistency)\nseparately for each of their subsystems.\n\nTag management falls into the same category, as some AWS subsystems rely on\nthe Resource and Tag Manager to handle tagging, and that means more eventual\nconsistency.  If you modify or remove a tag on a resource, it is often not\nrealized immediately (causing a situation where you cannot read what you\nwrote), so follow-up code that depends on that tag being modified or deleted\ncan fail.\n\nboto3-post-conditions hooks into [botocore](https://github.com/boto/botocore)\nin a manner similar to the simulation library [moto](https://github.com/spulec/moto)\nsuch that when a call completes that modifies something, the client verifies\nthat the change has been realized.  This does require a little bit of stack\ninspection, as the client and original call parameters are not passed through\nthe handle_event mechanism.\n\n## Theory of Operation\n\nEach boto3 call that makes a modification can be verified for liveness with\na subsequent read call.  The "after-call" boto3 event occurs after every call\nallowing code to plug into the request stream.  When the response indicates\na change is successful, post-conditions can be enforced by reading back the\nresource and ensuring the expected change is there.\n\nTake the Systems Manager (SSM) Parameter Store, for example.  As an eventually\nconsistent system, extra care must be taken to avoid some common pitfalls,\nsuch as:\n\n- Calling put_parameter immediately followed by get_parameter has no guarantee\n  of success.  In fact, it can raise a ParameterNotFound.\n- Calling delete_parameter immediately followed by get_parameter can still return\n  the parameter.\n\nboto3_post_conditions adds logic to ensure that the modifications you are making\nare realized by that subsystem\'s control plane before returning control to the\ncaller.  For example with put_parameter, boto3_post_conditions will ensure that:\n\n- The parameter can be read (with GetParameter) to handle the case of a novel\n  parameter being added.\n- The parameter\'s tags can be read, should the Tags be set during creation.\n  The resource and tag manager is a separate subsystem which adds even more\n  delays to realization.\n\nEvery modification or deletion has a post-condition remedy that can ensure\nthe vast majority of these cases are eliminated, and virtually removing retry\nlogic from your code and tests!  As proof, look at the recorded test for\n`test_ssm_integration`.\n',
    'author': 'James E. King III',
    'author_email': 'jking@apache.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/jeking3/boto3-post-conditions/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<3.11',
}


setup(**setup_kwargs)
