"""
This module is responsible for uploading files to the server
The URL at which we want to upload the file is a presigned URL
that is generated by the server and sent to the client.

The method should be as fast as possible, and should also show a progress bar
"""
import os
from tqdm import tqdm
from tqdm.utils import CallbackIOWrapper
from ..Request import _Request
import requests
from dataclasses import dataclass
from typing import Callable, Dict, Any
import logging
from .sha import calculate_sha1_hash_base64

logger = logging.getLogger(__name__)


@dataclass
class ParamsArgs:
    sha1hash: str
    file_name: str


@dataclass
class FileUploadResponse:
    sha1_hash: str
    file_size: int
    presigned_endpoint_response: Any


class FileUploader:
    def __init__(self, request: _Request):
        self.request = request

    def upload_file(
        self,
        file_path: str,
        presigned_endpoint_url: str,
        construct_params: Callable[[ParamsArgs], Dict[str, Any]],
        request_method: str = "POST",
    ):
        file_name = os.path.basename(file_path)

        with open(file_path, "rb") as f:
            sha1_hash, file_size = calculate_sha1_hash_base64(f)
            # Seek back to the beginning of the file
            f.seek(0)

            params = construct_params(
                ParamsArgs(sha1hash=sha1_hash, file_name=file_name)
            )
            res = self.request._make_request(
                request_method, presigned_endpoint_url, params=params
            )
            presigned_url = res["url"]

            # Using tqdm as an iterable to show progress bar while streaming the entire file
            with tqdm(
                total=file_size, unit="B", unit_scale=True, unit_divisor=1024
            ) as t:
                wrapped_file = CallbackIOWrapper(t.update, f, "read")
                response = requests.put(
                    presigned_url,
                    data=wrapped_file,
                    headers={
                        "x-amz-checksum-sha1": sha1_hash,
                    },
                    verify=False
                )
                if not response.ok or response.status_code != 200:
                    logger.error(
                        f"Failed to upload {file_name}. Server responded with: {response.status_code}. {response.text}"
                    )

        response.raise_for_status()
        return FileUploadResponse(
            sha1_hash=sha1_hash,
            file_size=file_size,
            presigned_endpoint_response=res,
        )
