"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheQueryStringBehavior = exports.CacheHeaderBehavior = exports.CacheCookieBehavior = exports.CachePolicy = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Cache Policy configuration.
 *
 * @stability stable
 * @resource AWS::CloudFront::CachePolicy
 */
class CachePolicy extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.cachePolicyName,
        });
        const cachePolicyName = (_e = props.cachePolicyName) !== null && _e !== void 0 ? _e : core_1.Names.uniqueId(this);
        if (!core_1.Token.isUnresolved(cachePolicyName) && !cachePolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'cachePolicyName' can only include '-', '_', and alphanumeric characters, got: '${props.cachePolicyName}'`);
        }
        const minTtl = ((_f = props.minTtl) !== null && _f !== void 0 ? _f : core_1.Duration.seconds(0)).toSeconds();
        const defaultTtl = Math.max(((_g = props.defaultTtl) !== null && _g !== void 0 ? _g : core_1.Duration.days(1)).toSeconds(), minTtl);
        const maxTtl = Math.max(((_h = props.maxTtl) !== null && _h !== void 0 ? _h : core_1.Duration.days(365)).toSeconds(), defaultTtl);
        const resource = new cloudfront_generated_1.CfnCachePolicy(this, 'Resource', {
            cachePolicyConfig: {
                name: cachePolicyName,
                comment: props.comment,
                minTtl,
                maxTtl,
                defaultTtl,
                parametersInCacheKeyAndForwardedToOrigin: this.renderCacheKey(props),
            },
        });
        this.cachePolicyId = resource.ref;
    }
    /**
     * Imports a Cache Policy from its id.
     *
     * @stability stable
     */
    static fromCachePolicyId(scope, id, cachePolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.cachePolicyId = cachePolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed cache policy. */
    static fromManagedCachePolicy(managedCachePolicyId) {
        return new class {
            constructor() {
                this.cachePolicyId = managedCachePolicyId;
            }
        }();
    }
    renderCacheKey(props) {
        var _e, _f, _g, _h, _j;
        const cookies = (_e = props.cookieBehavior) !== null && _e !== void 0 ? _e : CacheCookieBehavior.none();
        const headers = (_f = props.headerBehavior) !== null && _f !== void 0 ? _f : CacheHeaderBehavior.none();
        const queryStrings = (_g = props.queryStringBehavior) !== null && _g !== void 0 ? _g : CacheQueryStringBehavior.none();
        return {
            cookiesConfig: {
                cookieBehavior: cookies.behavior,
                cookies: cookies.cookies,
            },
            headersConfig: {
                headerBehavior: headers.behavior,
                headers: headers.headers,
            },
            enableAcceptEncodingGzip: (_h = props.enableAcceptEncodingGzip) !== null && _h !== void 0 ? _h : false,
            enableAcceptEncodingBrotli: (_j = props.enableAcceptEncodingBrotli) !== null && _j !== void 0 ? _j : false,
            queryStringsConfig: {
                queryStringBehavior: queryStrings.behavior,
                queryStrings: queryStrings.queryStrings,
            },
        };
    }
}
exports.CachePolicy = CachePolicy;
_a = JSII_RTTI_SYMBOL_1;
CachePolicy[_a] = { fqn: "@aws-cdk/aws-cloudfront.CachePolicy", version: "1.94.1" };
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 *
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 *
 * @stability stable
 */
CachePolicy.CACHING_OPTIMIZED = CachePolicy.fromManagedCachePolicy('658327ea-f89d-4fab-a63d-7e88639e58f6');
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 *
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 * Disables cache compression.
 *
 * @stability stable
 */
CachePolicy.CACHING_OPTIMIZED_FOR_UNCOMPRESSED_OBJECTS = CachePolicy.fromManagedCachePolicy('b2884449-e4de-46a7-ac36-70bc7f1ddd6d');
/**
 * Disables caching.
 *
 * This policy is useful for dynamic content and for requests that are not cacheable.
 *
 * @stability stable
 */
CachePolicy.CACHING_DISABLED = CachePolicy.fromManagedCachePolicy('4135ea2d-6df8-44a3-9df3-4b5a84be39ad');
/**
 * Designed for use with an origin that is an AWS Elemental MediaPackage endpoint.
 *
 * @stability stable
 */
CachePolicy.ELEMENTAL_MEDIA_PACKAGE = CachePolicy.fromManagedCachePolicy('08627262-05a9-4f76-9ded-b50ca2e3a84f');
/**
 * Determines whether any cookies in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class CacheCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * Cookies in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none() { return new CacheCookieBehavior('none'); }
    /**
     * All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new CacheCookieBehavior('all'); }
    /**
     * Only the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new CacheCookieBehavior('whitelist', cookies);
    }
    /**
     * All cookies except the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static denyList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to deny must be provided');
        }
        return new CacheCookieBehavior('allExcept', cookies);
    }
}
exports.CacheCookieBehavior = CacheCookieBehavior;
_b = JSII_RTTI_SYMBOL_1;
CacheCookieBehavior[_b] = { fqn: "@aws-cdk/aws-cloudfront.CacheCookieBehavior", version: "1.94.1" };
/**
 * Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class CacheHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /**
     * HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none() { return new CacheHeaderBehavior('none'); }
    /**
     * Listed headers are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        if (headers.length > 10) {
            throw new Error(`Maximum allowed headers in Cache Policy is 10; got ${headers.length}.`);
        }
        return new CacheHeaderBehavior('whitelist', headers);
    }
}
exports.CacheHeaderBehavior = CacheHeaderBehavior;
_c = JSII_RTTI_SYMBOL_1;
CacheHeaderBehavior[_c] = { fqn: "@aws-cdk/aws-cloudfront.CacheHeaderBehavior", version: "1.94.1" };
/**
 * Determines whether any URL query strings in viewer requests are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 *
 * @stability stable
 */
class CacheQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * Query strings in viewer requests are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static none() { return new CacheQueryStringBehavior('none'); }
    /**
     * All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static all() { return new CacheQueryStringBehavior('all'); }
    /**
     * Only the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new CacheQueryStringBehavior('whitelist', queryStrings);
    }
    /**
     * All query strings except the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     *
     * @stability stable
     */
    static denyList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to deny must be provided');
        }
        return new CacheQueryStringBehavior('allExcept', queryStrings);
    }
}
exports.CacheQueryStringBehavior = CacheQueryStringBehavior;
_d = JSII_RTTI_SYMBOL_1;
CacheQueryStringBehavior[_d] = { fqn: "@aws-cdk/aws-cloudfront.CacheQueryStringBehavior", version: "1.94.1" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FjaGUtcG9saWN5LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY2FjaGUtcG9saWN5LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsd0NBQWlFO0FBRWpFLGlFQUF3RDs7Ozs7OztBQXNGeEQsTUFBYSxXQUFZLFNBQVEsZUFBUTs7OztJQWtDdkMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUEwQixFQUFFOztRQUNwRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNmLFlBQVksRUFBRSxLQUFLLENBQUMsZUFBZTtTQUNwQyxDQUFDLENBQUM7UUFFSCxNQUFNLGVBQWUsU0FBRyxLQUFLLENBQUMsZUFBZSxtQ0FBSSxZQUFLLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RFLElBQUksQ0FBQyxZQUFLLENBQUMsWUFBWSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUMvRSxNQUFNLElBQUksS0FBSyxDQUFDLG1GQUFtRixLQUFLLENBQUMsZUFBZSxHQUFHLENBQUMsQ0FBQztTQUM5SDtRQUVELE1BQU0sTUFBTSxHQUFHLE9BQUMsS0FBSyxDQUFDLE1BQU0sbUNBQUksZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsRUFBRSxDQUFDO1FBQ2pFLE1BQU0sVUFBVSxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUMsT0FBQyxLQUFLLENBQUMsVUFBVSxtQ0FBSSxlQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxFQUFFLEVBQUUsTUFBTSxDQUFDLENBQUM7UUFDeEYsTUFBTSxNQUFNLEdBQUcsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUFDLEtBQUssQ0FBQyxNQUFNLG1DQUFJLGVBQVEsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRSxVQUFVLENBQUMsQ0FBQztRQUV0RixNQUFNLFFBQVEsR0FBRyxJQUFJLHFDQUFjLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwRCxpQkFBaUIsRUFBRTtnQkFDakIsSUFBSSxFQUFFLGVBQWU7Z0JBQ3JCLE9BQU8sRUFBRSxLQUFLLENBQUMsT0FBTztnQkFDdEIsTUFBTTtnQkFDTixNQUFNO2dCQUNOLFVBQVU7Z0JBQ1Ysd0NBQXdDLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxLQUFLLENBQUM7YUFDckU7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsYUFBYSxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7SUFDcEMsQ0FBQzs7Ozs7O0lBekNNLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxhQUFxQjtRQUNqRixPQUFPLElBQUksS0FBTSxTQUFRLGVBQVE7WUFBdEI7O2dCQUNPLGtCQUFhLEdBQUcsYUFBYSxDQUFDO1lBQ2hELENBQUM7U0FBQSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNmLENBQUM7SUFFRCw0Q0FBNEM7SUFDcEMsTUFBTSxDQUFDLHNCQUFzQixDQUFDLG9CQUE0QjtRQUNoRSxPQUFPLElBQUk7WUFBQTtnQkFDTyxrQkFBYSxHQUFHLG9CQUFvQixDQUFDO1lBQ3ZELENBQUM7U0FBQSxFQUFFLENBQUM7SUFDTixDQUFDO0lBZ0NPLGNBQWMsQ0FBQyxLQUF1Qjs7UUFDNUMsTUFBTSxPQUFPLFNBQUcsS0FBSyxDQUFDLGNBQWMsbUNBQUksbUJBQW1CLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDbkUsTUFBTSxPQUFPLFNBQUcsS0FBSyxDQUFDLGNBQWMsbUNBQUksbUJBQW1CLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDbkUsTUFBTSxZQUFZLFNBQUcsS0FBSyxDQUFDLG1CQUFtQixtQ0FBSSx3QkFBd0IsQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUVsRixPQUFPO1lBQ0wsYUFBYSxFQUFFO2dCQUNiLGNBQWMsRUFBRSxPQUFPLENBQUMsUUFBUTtnQkFDaEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPO2FBQ3pCO1lBQ0QsYUFBYSxFQUFFO2dCQUNiLGNBQWMsRUFBRSxPQUFPLENBQUMsUUFBUTtnQkFDaEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPO2FBQ3pCO1lBQ0Qsd0JBQXdCLFFBQUUsS0FBSyxDQUFDLHdCQUF3QixtQ0FBSSxLQUFLO1lBQ2pFLDBCQUEwQixRQUFFLEtBQUssQ0FBQywwQkFBMEIsbUNBQUksS0FBSztZQUNyRSxrQkFBa0IsRUFBRTtnQkFDbEIsbUJBQW1CLEVBQUUsWUFBWSxDQUFDLFFBQVE7Z0JBQzFDLFlBQVksRUFBRSxZQUFZLENBQUMsWUFBWTthQUN4QztTQUNGLENBQUM7SUFDSixDQUFDOztBQW5GSCxrQ0FvRkM7Ozs7Ozs7Ozs7QUE5RXdCLDZCQUFpQixHQUFHLFdBQVcsQ0FBQyxzQkFBc0IsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDOzs7Ozs7Ozs7QUFNL0Ysc0RBQTBDLEdBQUcsV0FBVyxDQUFDLHNCQUFzQixDQUFDLHNDQUFzQyxDQUFDLENBQUM7Ozs7Ozs7O0FBRXhILDRCQUFnQixHQUFHLFdBQVcsQ0FBQyxzQkFBc0IsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDOzs7Ozs7QUFFOUYsbUNBQXVCLEdBQUcsV0FBVyxDQUFDLHNCQUFzQixDQUFDLHNDQUFzQyxDQUFDLENBQUM7Ozs7OztBQTBFOUgsTUFBYSxtQkFBbUI7SUFzQzlCLFlBQW9CLFFBQWdCLEVBQUUsT0FBa0I7UUFDdEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUM7UUFDekIsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7SUFDekIsQ0FBQzs7Ozs7O0lBcENNLE1BQU0sQ0FBQyxJQUFJLEtBQUssT0FBTyxJQUFJLG1CQUFtQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQzs7Ozs7O0lBS3pELE1BQU0sQ0FBQyxHQUFHLEtBQUssT0FBTyxJQUFJLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsQ0FBQzs7Ozs7O0lBS3ZELE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxPQUFpQjtRQUMxQyxJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztTQUNsRTtRQUNELE9BQU8sSUFBSSxtQkFBbUIsQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDdkQsQ0FBQzs7Ozs7O0lBTU0sTUFBTSxDQUFDLFFBQVEsQ0FBQyxHQUFHLE9BQWlCO1FBQ3pDLElBQUksT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDeEIsTUFBTSxJQUFJLEtBQUssQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1NBQ2pFO1FBQ0QsT0FBTyxJQUFJLG1CQUFtQixDQUFDLFdBQVcsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUN2RCxDQUFDOztBQS9CSCxrREEwQ0M7Ozs7Ozs7O0FBS0QsTUFBYSxtQkFBbUI7SUFtQjlCLFlBQW9CLFFBQWdCLEVBQUUsT0FBa0I7UUFDdEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxRQUFRLENBQUM7UUFDekIsSUFBSSxDQUFDLE9BQU8sR0FBRyxPQUFPLENBQUM7SUFDekIsQ0FBQzs7Ozs7O0lBcEJNLE1BQU0sQ0FBQyxJQUFJLEtBQUssT0FBTyxJQUFJLG1CQUFtQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQzs7Ozs7O0lBRXpELE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxPQUFpQjtRQUMxQyxJQUFJLE9BQU8sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hCLE1BQU0sSUFBSSxLQUFLLENBQUMsK0NBQStDLENBQUMsQ0FBQztTQUNsRTtRQUNELElBQUksT0FBTyxDQUFDLE1BQU0sR0FBRyxFQUFFLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsT0FBTyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7U0FDMUY7UUFDRCxPQUFPLElBQUksbUJBQW1CLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3ZELENBQUM7O0FBWkgsa0RBdUJDOzs7Ozs7OztBQU1ELE1BQWEsd0JBQXdCO0lBc0NuQyxZQUFvQixRQUFnQixFQUFFLFlBQXVCO1FBQzNELElBQUksQ0FBQyxRQUFRLEdBQUcsUUFBUSxDQUFDO1FBQ3pCLElBQUksQ0FBQyxZQUFZLEdBQUcsWUFBWSxDQUFDO0lBQ25DLENBQUM7Ozs7OztJQXBDTSxNQUFNLENBQUMsSUFBSSxLQUFLLE9BQU8sSUFBSSx3QkFBd0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7OztJQUs5RCxNQUFNLENBQUMsR0FBRyxLQUFLLE9BQU8sSUFBSSx3QkFBd0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7Ozs7OztJQUs1RCxNQUFNLENBQUMsU0FBUyxDQUFDLEdBQUcsWUFBc0I7UUFDL0MsSUFBSSxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7U0FDeEU7UUFDRCxPQUFPLElBQUksd0JBQXdCLENBQUMsV0FBVyxFQUFFLFlBQVksQ0FBQyxDQUFDO0lBQ2pFLENBQUM7Ozs7OztJQU1NLE1BQU0sQ0FBQyxRQUFRLENBQUMsR0FBRyxZQUFzQjtRQUM5QyxJQUFJLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMsb0RBQW9ELENBQUMsQ0FBQztTQUN2RTtRQUNELE9BQU8sSUFBSSx3QkFBd0IsQ0FBQyxXQUFXLEVBQUUsWUFBWSxDQUFDLENBQUM7SUFDakUsQ0FBQzs7QUEvQkgsNERBMENDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgRHVyYXRpb24sIE5hbWVzLCBSZXNvdXJjZSwgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuQ2FjaGVQb2xpY3kgfSBmcm9tICcuL2Nsb3VkZnJvbnQuZ2VuZXJhdGVkJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIElDYWNoZVBvbGljeSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjYWNoZVBvbGljeUlkOiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQ2FjaGVQb2xpY3lQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjYWNoZVBvbGljeU5hbWU/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNvbW1lbnQ/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZGVmYXVsdFR0bD86IER1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBtaW5UdGw/OiBEdXJhdGlvbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbWF4VHRsPzogRHVyYXRpb247XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjb29raWVCZWhhdmlvcj86IENhY2hlQ29va2llQmVoYXZpb3I7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaGVhZGVyQmVoYXZpb3I/OiBDYWNoZUhlYWRlckJlaGF2aW9yO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHF1ZXJ5U3RyaW5nQmVoYXZpb3I/OiBDYWNoZVF1ZXJ5U3RyaW5nQmVoYXZpb3I7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5hYmxlQWNjZXB0RW5jb2RpbmdHemlwPzogYm9vbGVhbjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5hYmxlQWNjZXB0RW5jb2RpbmdCcm90bGk/OiBib29sZWFuO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQ2FjaGVQb2xpY3kgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElDYWNoZVBvbGljeSB7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBDQUNISU5HX09QVElNSVpFRCA9IENhY2hlUG9saWN5LmZyb21NYW5hZ2VkQ2FjaGVQb2xpY3koJzY1ODMyN2VhLWY4OWQtNGZhYi1hNjNkLTdlODg2MzllNThmNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENBQ0hJTkdfT1BUSU1JWkVEX0ZPUl9VTkNPTVBSRVNTRURfT0JKRUNUUyA9IENhY2hlUG9saWN5LmZyb21NYW5hZ2VkQ2FjaGVQb2xpY3koJ2IyODg0NDQ5LWU0ZGUtNDZhNy1hYzM2LTcwYmM3ZjFkZGQ2ZCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IENBQ0hJTkdfRElTQUJMRUQgPSBDYWNoZVBvbGljeS5mcm9tTWFuYWdlZENhY2hlUG9saWN5KCc0MTM1ZWEyZC02ZGY4LTQ0YTMtOWRmMy00YjVhODRiZTM5YWQnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyByZWFkb25seSBFTEVNRU5UQUxfTUVESUFfUEFDS0FHRSA9IENhY2hlUG9saWN5LmZyb21NYW5hZ2VkQ2FjaGVQb2xpY3koJzA4NjI3MjYyLTA1YTktNGY3Ni05ZGVkLWI1MGNhMmUzYTg0ZicpO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGZyb21DYWNoZVBvbGljeUlkKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGNhY2hlUG9saWN5SWQ6IHN0cmluZyk6IElDYWNoZVBvbGljeSB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUNhY2hlUG9saWN5IHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBjYWNoZVBvbGljeUlkID0gY2FjaGVQb2xpY3lJZDtcbiAgICB9KHNjb3BlLCBpZCk7XG4gIH1cblxuICAvKiogVXNlIGFuIGV4aXN0aW5nIG1hbmFnZWQgY2FjaGUgcG9saWN5LiAqL1xuICBwcml2YXRlIHN0YXRpYyBmcm9tTWFuYWdlZENhY2hlUG9saWN5KG1hbmFnZWRDYWNoZVBvbGljeUlkOiBzdHJpbmcpOiBJQ2FjaGVQb2xpY3kge1xuICAgIHJldHVybiBuZXcgY2xhc3MgaW1wbGVtZW50cyBJQ2FjaGVQb2xpY3kge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGNhY2hlUG9saWN5SWQgPSBtYW5hZ2VkQ2FjaGVQb2xpY3lJZDtcbiAgICB9KCk7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgY2FjaGVQb2xpY3lJZDogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDYWNoZVBvbGljeVByb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuY2FjaGVQb2xpY3lOYW1lLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY2FjaGVQb2xpY3lOYW1lID0gcHJvcHMuY2FjaGVQb2xpY3lOYW1lID8/IE5hbWVzLnVuaXF1ZUlkKHRoaXMpO1xuICAgIGlmICghVG9rZW4uaXNVbnJlc29sdmVkKGNhY2hlUG9saWN5TmFtZSkgJiYgIWNhY2hlUG9saWN5TmFtZS5tYXRjaCgvXltcXHctXSskL2kpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYCdjYWNoZVBvbGljeU5hbWUnIGNhbiBvbmx5IGluY2x1ZGUgJy0nLCAnXycsIGFuZCBhbHBoYW51bWVyaWMgY2hhcmFjdGVycywgZ290OiAnJHtwcm9wcy5jYWNoZVBvbGljeU5hbWV9J2ApO1xuICAgIH1cblxuICAgIGNvbnN0IG1pblR0bCA9IChwcm9wcy5taW5UdGwgPz8gRHVyYXRpb24uc2Vjb25kcygwKSkudG9TZWNvbmRzKCk7XG4gICAgY29uc3QgZGVmYXVsdFR0bCA9IE1hdGgubWF4KChwcm9wcy5kZWZhdWx0VHRsID8/IER1cmF0aW9uLmRheXMoMSkpLnRvU2Vjb25kcygpLCBtaW5UdGwpO1xuICAgIGNvbnN0IG1heFR0bCA9IE1hdGgubWF4KChwcm9wcy5tYXhUdGwgPz8gRHVyYXRpb24uZGF5cygzNjUpKS50b1NlY29uZHMoKSwgZGVmYXVsdFR0bCk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5DYWNoZVBvbGljeSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBjYWNoZVBvbGljeUNvbmZpZzoge1xuICAgICAgICBuYW1lOiBjYWNoZVBvbGljeU5hbWUsXG4gICAgICAgIGNvbW1lbnQ6IHByb3BzLmNvbW1lbnQsXG4gICAgICAgIG1pblR0bCxcbiAgICAgICAgbWF4VHRsLFxuICAgICAgICBkZWZhdWx0VHRsLFxuICAgICAgICBwYXJhbWV0ZXJzSW5DYWNoZUtleUFuZEZvcndhcmRlZFRvT3JpZ2luOiB0aGlzLnJlbmRlckNhY2hlS2V5KHByb3BzKSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICB0aGlzLmNhY2hlUG9saWN5SWQgPSByZXNvdXJjZS5yZWY7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlckNhY2hlS2V5KHByb3BzOiBDYWNoZVBvbGljeVByb3BzKTogQ2ZuQ2FjaGVQb2xpY3kuUGFyYW1ldGVyc0luQ2FjaGVLZXlBbmRGb3J3YXJkZWRUb09yaWdpblByb3BlcnR5IHtcbiAgICBjb25zdCBjb29raWVzID0gcHJvcHMuY29va2llQmVoYXZpb3IgPz8gQ2FjaGVDb29raWVCZWhhdmlvci5ub25lKCk7XG4gICAgY29uc3QgaGVhZGVycyA9IHByb3BzLmhlYWRlckJlaGF2aW9yID8/IENhY2hlSGVhZGVyQmVoYXZpb3Iubm9uZSgpO1xuICAgIGNvbnN0IHF1ZXJ5U3RyaW5ncyA9IHByb3BzLnF1ZXJ5U3RyaW5nQmVoYXZpb3IgPz8gQ2FjaGVRdWVyeVN0cmluZ0JlaGF2aW9yLm5vbmUoKTtcblxuICAgIHJldHVybiB7XG4gICAgICBjb29raWVzQ29uZmlnOiB7XG4gICAgICAgIGNvb2tpZUJlaGF2aW9yOiBjb29raWVzLmJlaGF2aW9yLFxuICAgICAgICBjb29raWVzOiBjb29raWVzLmNvb2tpZXMsXG4gICAgICB9LFxuICAgICAgaGVhZGVyc0NvbmZpZzoge1xuICAgICAgICBoZWFkZXJCZWhhdmlvcjogaGVhZGVycy5iZWhhdmlvcixcbiAgICAgICAgaGVhZGVyczogaGVhZGVycy5oZWFkZXJzLFxuICAgICAgfSxcbiAgICAgIGVuYWJsZUFjY2VwdEVuY29kaW5nR3ppcDogcHJvcHMuZW5hYmxlQWNjZXB0RW5jb2RpbmdHemlwID8/IGZhbHNlLFxuICAgICAgZW5hYmxlQWNjZXB0RW5jb2RpbmdCcm90bGk6IHByb3BzLmVuYWJsZUFjY2VwdEVuY29kaW5nQnJvdGxpID8/IGZhbHNlLFxuICAgICAgcXVlcnlTdHJpbmdzQ29uZmlnOiB7XG4gICAgICAgIHF1ZXJ5U3RyaW5nQmVoYXZpb3I6IHF1ZXJ5U3RyaW5ncy5iZWhhdmlvcixcbiAgICAgICAgcXVlcnlTdHJpbmdzOiBxdWVyeVN0cmluZ3MucXVlcnlTdHJpbmdzLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBDYWNoZUNvb2tpZUJlaGF2aW9yIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBub25lKCkgeyByZXR1cm4gbmV3IENhY2hlQ29va2llQmVoYXZpb3IoJ25vbmUnKTsgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgYWxsKCkgeyByZXR1cm4gbmV3IENhY2hlQ29va2llQmVoYXZpb3IoJ2FsbCcpOyB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbG93TGlzdCguLi5jb29raWVzOiBzdHJpbmdbXSkge1xuICAgIGlmIChjb29raWVzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgY29va2llIHRvIGFsbG93IG11c3QgYmUgcHJvdmlkZWQnKTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBDYWNoZUNvb2tpZUJlaGF2aW9yKCd3aGl0ZWxpc3QnLCBjb29raWVzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgZGVueUxpc3QoLi4uY29va2llczogc3RyaW5nW10pIHtcbiAgICBpZiAoY29va2llcy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIGNvb2tpZSB0byBkZW55IG11c3QgYmUgcHJvdmlkZWQnKTtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBDYWNoZUNvb2tpZUJlaGF2aW9yKCdhbGxFeGNlcHQnLCBjb29raWVzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGJlaGF2aW9yOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgY29va2llcz86IHN0cmluZ1tdO1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoYmVoYXZpb3I6IHN0cmluZywgY29va2llcz86IHN0cmluZ1tdKSB7XG4gICAgdGhpcy5iZWhhdmlvciA9IGJlaGF2aW9yO1xuICAgIHRoaXMuY29va2llcyA9IGNvb2tpZXM7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBDYWNoZUhlYWRlckJlaGF2aW9yIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBub25lKCkgeyByZXR1cm4gbmV3IENhY2hlSGVhZGVyQmVoYXZpb3IoJ25vbmUnKTsgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbG93TGlzdCguLi5oZWFkZXJzOiBzdHJpbmdbXSkge1xuICAgIGlmIChoZWFkZXJzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgaGVhZGVyIHRvIGFsbG93IG11c3QgYmUgcHJvdmlkZWQnKTtcbiAgICB9XG4gICAgaWYgKGhlYWRlcnMubGVuZ3RoID4gMTApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTWF4aW11bSBhbGxvd2VkIGhlYWRlcnMgaW4gQ2FjaGUgUG9saWN5IGlzIDEwOyBnb3QgJHtoZWFkZXJzLmxlbmd0aH0uYCk7XG4gICAgfVxuICAgIHJldHVybiBuZXcgQ2FjaGVIZWFkZXJCZWhhdmlvcignd2hpdGVsaXN0JywgaGVhZGVycyk7XG4gIH1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGJlaGF2aW9yOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGhlYWRlcnM/OiBzdHJpbmdbXTtcblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKGJlaGF2aW9yOiBzdHJpbmcsIGhlYWRlcnM/OiBzdHJpbmdbXSkge1xuICAgIHRoaXMuYmVoYXZpb3IgPSBiZWhhdmlvcjtcbiAgICB0aGlzLmhlYWRlcnMgPSBoZWFkZXJzO1xuICB9XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgQ2FjaGVRdWVyeVN0cmluZ0JlaGF2aW9yIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBub25lKCkgeyByZXR1cm4gbmV3IENhY2hlUXVlcnlTdHJpbmdCZWhhdmlvcignbm9uZScpOyB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBhbGwoKSB7IHJldHVybiBuZXcgQ2FjaGVRdWVyeVN0cmluZ0JlaGF2aW9yKCdhbGwnKTsgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGFsbG93TGlzdCguLi5xdWVyeVN0cmluZ3M6IHN0cmluZ1tdKSB7XG4gICAgaWYgKHF1ZXJ5U3RyaW5ncy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIHF1ZXJ5IHN0cmluZyB0byBhbGxvdyBtdXN0IGJlIHByb3ZpZGVkJyk7XG4gICAgfVxuICAgIHJldHVybiBuZXcgQ2FjaGVRdWVyeVN0cmluZ0JlaGF2aW9yKCd3aGl0ZWxpc3QnLCBxdWVyeVN0cmluZ3MpO1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgc3RhdGljIGRlbnlMaXN0KC4uLnF1ZXJ5U3RyaW5nczogc3RyaW5nW10pIHtcbiAgICBpZiAocXVlcnlTdHJpbmdzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgcXVlcnkgc3RyaW5nIHRvIGRlbnkgbXVzdCBiZSBwcm92aWRlZCcpO1xuICAgIH1cbiAgICByZXR1cm4gbmV3IENhY2hlUXVlcnlTdHJpbmdCZWhhdmlvcignYWxsRXhjZXB0JywgcXVlcnlTdHJpbmdzKTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBiZWhhdmlvcjogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHF1ZXJ5U3RyaW5ncz86IHN0cmluZ1tdO1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3IoYmVoYXZpb3I6IHN0cmluZywgcXVlcnlTdHJpbmdzPzogc3RyaW5nW10pIHtcbiAgICB0aGlzLmJlaGF2aW9yID0gYmVoYXZpb3I7XG4gICAgdGhpcy5xdWVyeVN0cmluZ3MgPSBxdWVyeVN0cmluZ3M7XG4gIH1cbn1cbiJdfQ==