import tensorlayerx as tlx
from tensorlayerx import nn
import math
from ..resnet.resnet50 import ResNet50
import numpy as np
import cv2


class ArcMarginPenaltyLogists(nn.Module):
    """ArcMarginPenaltyLogists"""

    def __init__(self, num_classes, margin=0.5, logist_scale=64, **kwargs):
        super(ArcMarginPenaltyLogists, self).__init__(**kwargs)
        self.num_classes = num_classes
        self.margin = margin
        self.logist_scale = logist_scale

        self.one_hot = tlx.OneHot(self.num_classes)

    def build(self, inputs_shape):
        self.w = self._get_weights(
            "weights", shape=[int(inputs_shape[-1]), self.num_classes])
        self.cos_m = tlx.identity(math.cos(self.margin))
        self.sin_m = tlx.identity(math.sin(self.margin))
        self.th = tlx.identity(math.cos(math.pi - self.margin))
        self.mm = tlx.multiply(self.sin_m, self.margin)

    def forward(self, embds, labels):
        normed_embds = tlx.l2_normalize(embds, axis=1)
        normed_w = tlx.l2_normalize(self.w, axis=0)

        cos_t = tlx.matmul(normed_embds, normed_w)
        sin_t = tlx.sqrt(1. - cos_t ** 2)

        cos_mt = tlx.subtract(cos_t * self.cos_m, sin_t * self.sin_m)

        cos_mt = tlx.where(cos_t > self.th, cos_mt, cos_t - self.mm)

        mask = self.one_hot(tlx.cast(labels, tlx.int32))

        logists = tlx.where(mask == 1., cos_mt, cos_t)
        logists = tlx.multiply(logists, self.logist_scale)

        return logists


class ArcHead(nn.Module):
    def __init__(self, num_classes, margin=0.5, logist_scale=64, name='ArcHead'):
        super(ArcHead, self).__init__(name=name)
        self.arc_head = ArcMarginPenaltyLogists(num_classes=num_classes,
                                                margin=margin,
                                                logist_scale=logist_scale)

    def forward(self, x_in, y_in):
        return self.arc_head(x_in, y_in)


class NormHead(nn.Module):
    def __init__(self, num_classes, w_decay=5e-4, name='NormHead'):
        super(NormHead, self).__init__(name=name)
        self.dense = tlx.nn.Linear(out_features=num_classes)

    def forward(self, inputs):
        return self.dense(inputs)


class ArcFace(nn.Module):
    def __init__(self, size=None, embd_shape=512, channels=3, name='arcface'):
        """
        :param size: (:obj:`int`, `optional`):
            input size for build model.
        :param embd_shape: (:obj:`int`, `optional`, defaults to 512):
            Number of hidden in the dense.
        :param channels: (:obj:`int`, `optional`, defaults to 3):
            channels for build model.
        """
        super(ArcFace, self).__init__(name=name)

        self.backbone = ResNet50(None, use_preprocess=False)

        self.bn = nn.BatchNorm(0.99, epsilon=1.001e-5, name="bn")
        self.dropout = nn.Dropout(0.5)
        self.flatten = nn.Flatten()
        self.dense = nn.Linear(out_features=embd_shape, name="dense")
        self.bn2 = nn.BatchNorm(0.99, epsilon=1.001e-5, name="bn2")

        self.size = size

        if size is not None:
            self.build(inputs_shape=[2, size, size, channels])

    def build(self, inputs_shape):
        ones = tlx.ones(inputs_shape)
        _ = self(ones)

    def forward(self, inputs):
        x = self.backbone(inputs)

        x = self.bn(x)
        x = self.dropout(x)
        x = self.flatten(x)
        x = self.dense(x)
        x = self.bn2(x)

        return x


def load_ckpt_weight(model, file_path):
    import tensorflow as tf

    weight_name_dict = {'conv1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-0/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-0/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-1/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-1/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-1/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-1/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-2/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-2/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-3/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-3/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-3/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-3/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-4/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-4/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-5/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-5/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-5/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-5/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_0_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-6/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_0_conv/bias:0': 'layer_with_weights-0/layer_with_weights-6/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-7/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-7/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_0_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-8/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_0_bn/beta:0': 'layer_with_weights-0/layer_with_weights-8/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_0_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-8/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_0_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-8/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-9/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-8/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-9/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block1_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-9/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-10/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-10/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-11/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-11/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-11/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-11/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-12/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-12/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-13/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-13/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-13/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-13/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-14/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-14/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-15/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-15/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-15/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block2_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-15/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-16/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-16/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-17/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-17/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-17/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-17/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-18/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-18/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-19/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-19/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-19/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-19/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-20/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-20/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-21/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-21/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-21/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv2_block3_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-21/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-22/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-22/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-23/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-23/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-23/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-23/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-24/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-24/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-25/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-25/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-25/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-25/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_0_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-26/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_0_conv/bias:0': 'layer_with_weights-0/layer_with_weights-26/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-27/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-27/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_0_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-28/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_0_bn/beta:0': 'layer_with_weights-0/layer_with_weights-28/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_0_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-28/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_0_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-28/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-29/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-28/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-29/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block1_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-29/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-30/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-30/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-31/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-31/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-31/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-31/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-32/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-32/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-33/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-33/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-33/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-33/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-34/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-34/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-35/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-35/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-35/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block2_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-35/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-36/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-36/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-37/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-37/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-37/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-37/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-38/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-38/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-39/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-39/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-39/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-39/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-40/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-40/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-41/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-41/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-41/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block3_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-41/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-42/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-42/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-43/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-43/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-43/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-43/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-44/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-44/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-45/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-45/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-45/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-45/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-46/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-46/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-47/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-47/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-47/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv3_block4_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-47/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-48/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-48/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-49/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-49/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-49/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-49/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-50/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-50/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-51/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-51/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-51/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-51/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_0_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-52/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_0_conv/bias:0': 'layer_with_weights-0/layer_with_weights-52/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-53/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-53/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_0_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-54/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_0_bn/beta:0': 'layer_with_weights-0/layer_with_weights-54/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_0_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-54/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_0_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-54/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-55/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-54/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-55/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block1_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-55/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-56/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-56/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-57/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-57/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-57/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-57/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-58/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-58/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-59/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-59/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-59/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-59/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-60/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-60/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-61/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-61/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-61/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block2_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-61/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-62/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-62/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-63/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-63/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-63/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-63/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-64/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-64/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-65/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-65/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-65/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-65/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-66/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-66/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-67/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-67/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-67/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block3_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-67/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-68/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-68/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-69/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-69/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-69/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-69/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-70/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-70/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-71/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-71/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-71/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-71/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-72/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-72/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-73/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-73/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-73/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block4_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-73/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-74/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-74/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-75/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-75/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-75/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-75/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-76/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-76/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-77/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-77/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-77/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-77/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-78/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-78/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-79/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-79/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-79/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block5_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-79/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-80/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-80/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-81/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-81/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-81/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-81/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-82/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-82/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-83/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-83/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-83/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-83/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-84/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-84/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-85/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-85/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-85/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv4_block6_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-85/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-86/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-86/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-87/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-87/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-87/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-87/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-88/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-88/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-89/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-89/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-89/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-89/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_0_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-90/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_0_conv/bias:0': 'layer_with_weights-0/layer_with_weights-90/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-91/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-91/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_0_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-92/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_0_bn/beta:0': 'layer_with_weights-0/layer_with_weights-92/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_0_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-92/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_0_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-92/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-93/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-92/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-93/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block1_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-93/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-94/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-94/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-95/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-95/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-95/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-95/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-96/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-96/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-97/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-97/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-97/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-97/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-98/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-98/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-99/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-99/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-99/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block2_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-99/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_1_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-100/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_1_conv/bias:0': 'layer_with_weights-0/layer_with_weights-100/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_1_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-101/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_1_bn/beta:0': 'layer_with_weights-0/layer_with_weights-101/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_1_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-101/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_1_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-101/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_2_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-102/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_2_conv/bias:0': 'layer_with_weights-0/layer_with_weights-102/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_2_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-103/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_2_bn/beta:0': 'layer_with_weights-0/layer_with_weights-103/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_2_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-103/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_2_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-103/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_3_conv/kernel:0': 'layer_with_weights-0/layer_with_weights-104/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_3_conv/bias:0': 'layer_with_weights-0/layer_with_weights-104/bias/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_3_bn/gamma:0': 'layer_with_weights-0/layer_with_weights-105/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_3_bn/beta:0': 'layer_with_weights-0/layer_with_weights-105/beta/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_3_bn/moving_mean:0': 'layer_with_weights-0/layer_with_weights-105/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'conv5_block3_3_bn/moving_variance:0': 'layer_with_weights-0/layer_with_weights-105/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'bn/gamma:0': 'layer_with_weights-1/layer_with_weights-0/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'bn/beta:0': 'layer_with_weights-1/layer_with_weights-0/beta/.ATTRIBUTES/VARIABLE_VALUE', 'bn/moving_mean:0': 'layer_with_weights-1/layer_with_weights-0/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'bn/moving_variance:0': 'layer_with_weights-1/layer_with_weights-0/moving_variance/.ATTRIBUTES/VARIABLE_VALUE', 'dense/kernel:0': 'layer_with_weights-1/layer_with_weights-1/kernel/.ATTRIBUTES/VARIABLE_VALUE', 'dense/bias:0': 'layer_with_weights-1/layer_with_weights-1/bias/.ATTRIBUTES/VARIABLE_VALUE', 'bn2/gamma:0': 'layer_with_weights-1/layer_with_weights-2/gamma/.ATTRIBUTES/VARIABLE_VALUE', 'bn2/beta:0': 'layer_with_weights-1/layer_with_weights-2/beta/.ATTRIBUTES/VARIABLE_VALUE', 'bn2/moving_mean:0': 'layer_with_weights-1/layer_with_weights-2/moving_mean/.ATTRIBUTES/VARIABLE_VALUE', 'bn2/moving_variance:0': 'layer_with_weights-1/layer_with_weights-2/moving_variance/.ATTRIBUTES/VARIABLE_VALUE'}
    names = []
    for w in model.all_weights:
        name = w.name
        name = name.replace("filters:0", "kernel:0")
        name = name.replace("biases:0", "bias:0")
        name = name.replace("moving_var:0", "moving_variance:0")
        name = name.replace("dense/weights:0", "dense/kernel:0")
        name = weight_name_dict[name]
        array = tf.train.load_variable(file_path, name)
        names.append(name)
        w.assign(array)

    return model


def l2_norm(x, axis=1):
    """l2 norm"""
    norm = np.linalg.norm(x, axis=axis, keepdims=True)
    output = x / norm

    return output


def get_face_emb(face, arcface, size=112):
    img = cv2.resize(face, (size, size))
    img = img.astype(np.float32) / 255.
    if len(img.shape) == 3:
        img = np.expand_dims(img, 0)
    emb = l2_norm(arcface(img))
    return tlx.convert_to_numpy(emb)