"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const sfn = require("../lib");
describe('Task base', () => {
    let stack;
    let task;
    beforeEach(() => {
        // GIVEN
        stack = new cdk.Stack();
        task = new FakeTask(stack, 'my-task', {
            metrics: {
                metricPrefixPlural: '',
                metricPrefixSingular: '',
            },
        });
    });
    test('instantiate a concrete implementation with properties', () => {
        // WHEN
        task = new FakeTask(stack, 'my-exciting-task', {
            comment: 'my exciting task',
            heartbeat: cdk.Duration.seconds(10),
            timeout: cdk.Duration.minutes(10),
        });
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-exciting-task',
            States: {
                'my-exciting-task': {
                    End: true,
                    Type: 'Task',
                    Comment: 'my exciting task',
                    TimeoutSeconds: 600,
                    HeartbeatSeconds: 10,
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
            },
        });
    });
    test('add catch configuration', () => {
        // GIVEN
        const failure = new sfn.Fail(stack, 'failed', {
            error: 'DidNotWork',
            cause: 'We got stuck',
        });
        // WHEN
        task.addCatch(failure);
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    End: true,
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'failed',
                        }],
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
                'failed': {
                    Type: 'Fail',
                    Error: 'DidNotWork',
                    Cause: 'We got stuck',
                },
            },
        });
    });
    test('add retry configuration', () => {
        // WHEN
        task.addRetry({ errors: ['HTTPError'], maxAttempts: 2 })
            .addRetry(); // adds default retry
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    End: true,
                    Retry: [
                        {
                            ErrorEquals: ['HTTPError'],
                            MaxAttempts: 2,
                        },
                        {
                            ErrorEquals: ['States.ALL'],
                        },
                    ],
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
            },
        });
    });
    test('add a next state to the task in the chain', () => {
        // WHEN
        task.next(new sfn.Pass(stack, 'passState'));
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    Next: 'passState',
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
                'passState': { Type: 'Pass', End: true },
            },
        });
    });
    test('get named metric for this task', () => {
        // WHEN
        const metric = task.metric('my-metric');
        // THEN
        verifyMetric(metric, 'my-metric', 'Sum');
    });
    test('add metric for task state run time', () => {
        // WHEN
        const metric = task.metricRunTime();
        // THEN
        verifyMetric(metric, 'RunTime', 'Average');
    });
    test('add metric for task schedule time', () => {
        // WHEN
        const metric = task.metricScheduleTime();
        // THEN
        verifyMetric(metric, 'ScheduleTime', 'Average');
    });
    test('add metric for time between task being scheduled to closing', () => {
        // WHEN
        const metric = task.metricTime();
        // THEN
        verifyMetric(metric, 'Time', 'Average');
    });
    test('add metric for number of times the task is scheduled', () => {
        // WHEN
        const metric = task.metricScheduled();
        // THEN
        verifyMetric(metric, 'Scheduled', 'Sum');
    });
    test('add metric for number of times the task times out', () => {
        // WHEN
        const metric = task.metricTimedOut();
        // THEN
        verifyMetric(metric, 'TimedOut', 'Sum');
    });
    test('add metric for number of times the task was started', () => {
        // WHEN
        const metric = task.metricStarted();
        // THEN
        verifyMetric(metric, 'Started', 'Sum');
    });
    test('add metric for number of times the task succeeded', () => {
        // WHEN
        const metric = task.metricSucceeded();
        // THEN
        verifyMetric(metric, 'Succeeded', 'Sum');
    });
    test('add metric for number of times the task failed', () => {
        // WHEN
        const metric = task.metricFailed();
        // THEN
        verifyMetric(metric, 'Failed', 'Sum');
    });
    test('add metric for number of times the metrics heartbeat timed out', () => {
        // WHEN
        const metric = task.metricHeartbeatTimedOut();
        // THEN
        verifyMetric(metric, 'HeartbeatTimedOut', 'Sum');
    });
    test('metrics must be configured to use metric* APIs', () => {
        // GIVEN
        task = new FakeTask(stack, 'mytask', {});
        // THEN
        expect(() => {
            task.metricFailed();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricHeartbeatTimedOut();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricRunTime();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricScheduleTime();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricScheduled();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricStarted();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricSucceeded();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricTime();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricTimedOut();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
    });
});
function verifyMetric(metric, metricName, statistic) {
    expect(metric).toEqual(expect.objectContaining({
        namespace: 'AWS/States',
        metricName,
        statistic,
    }));
}
function render(sm) {
    return new cdk.Stack().resolve(new sfn.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
class FakeTask extends sfn.TaskStateBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.taskMetrics = props.metrics;
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: 'my-resource',
            Parameters: sfn.FieldUtils.renderObject({
                MyParameter: 'myParameter',
            }),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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