"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wait = exports.WaitTime = void 0;
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Represents the Wait state which delays a state machine from continuing for a specified time
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-wait-state.html
 */
class WaitTime {
    constructor(json) {
        this.json = json;
    }
    /**
     * Wait a fixed amount of time.
     */
    static duration(duration) { return new WaitTime({ Seconds: duration.toSeconds() }); }
    /**
     * Wait until the given ISO8601 timestamp
     *
     * @example 2016-03-14T01:59:00Z
     */
    static timestamp(timestamp) { return new WaitTime({ Timestamp: timestamp }); }
    /**
     * Wait for a number of seconds stored in the state object.
     *
     * @example $.waitSeconds
     */
    static secondsPath(path) { return new WaitTime({ SecondsPath: path }); }
    /**
     * Wait until a timestamp found in the state object.
     *
     * @example $.waitTimestamp
     */
    static timestampPath(path) { return new WaitTime({ TimestampPath: path }); }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.WaitTime = WaitTime;
/**
 * Define a Wait state in the state machine
 *
 * A Wait state can be used to delay execution of the state machine for a while.
 */
class Wait extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.time = props.time;
        this.endStates = [this];
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.WAIT,
            Comment: this.comment,
            ...this.time._json,
            ...this.renderNextEnd(),
        };
    }
}
exports.Wait = Wait;
//# sourceMappingURL=data:application/json;base64,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