"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Choice = void 0;
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Choice in the state machine
 *
 * A choice state can be used to make decisions based on the execution
 * state.
 */
class Choice extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [];
    }
    /**
     * If the given condition matches, continue execution with the given state
     */
    when(condition, next) {
        super.addChoice(condition, next.startState);
        return this;
    }
    /**
     * If none of the given conditions match, continue execution with the given state
     *
     * If no conditions match and no otherwise() has been given, an execution
     * error will be raised.
     */
    otherwise(def) {
        super.makeDefault(def.startState);
        return this;
    }
    /**
     * Return a Chain that contains all reachable end states from this Choice
     *
     * Use this to combine all possible choice paths back.
     */
    afterwards(options = {}) {
        const endStates = state_1.State.filterNextables(state_1.State.findReachableEndStates(this, { includeErrorHandlers: options.includeErrorHandlers }));
        if (options.includeOtherwise && this.defaultChoice) {
            throw new Error(`'includeOtherwise' set but Choice state ${this.stateId} already has an 'otherwise' transition`);
        }
        if (options.includeOtherwise) {
            endStates.push(new DefaultAsNext(this));
        }
        return chain_1.Chain.custom(this, endStates, this);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.CHOICE,
            Comment: this.comment,
            ...this.renderInputOutput(),
            ...this.renderChoices(),
        };
    }
}
exports.Choice = Choice;
/**
 * Adapter to make the .otherwise() transition settable through .next()
 */
class DefaultAsNext {
    constructor(choice) {
        this.choice = choice;
    }
    next(state) {
        this.choice.otherwise(state);
        return chain_1.Chain.sequence(this.choice, state);
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2hvaWNlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY2hvaWNlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLG9DQUFpQztBQUdqQyxxREFBaUQ7QUFDakQsbUNBQWdDO0FBa0NoQzs7Ozs7R0FLRztBQUNILE1BQWEsTUFBTyxTQUFRLGFBQUs7SUFHL0IsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxRQUFxQixFQUFFO1FBQ25FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBSFYsY0FBUyxHQUFnQixFQUFFLENBQUM7SUFJNUMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksSUFBSSxDQUFDLFNBQW9CLEVBQUUsSUFBZ0I7UUFDaEQsS0FBSyxDQUFDLFNBQVMsQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQzVDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksU0FBUyxDQUFDLEdBQWU7UUFDOUIsS0FBSyxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDbEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLFVBQVUsQ0FBQyxVQUE2QixFQUFFO1FBQy9DLE1BQU0sU0FBUyxHQUFHLGFBQUssQ0FBQyxlQUFlLENBQUMsYUFBSyxDQUFDLHNCQUFzQixDQUFDLElBQUksRUFBRSxFQUFFLG9CQUFvQixFQUFFLE9BQU8sQ0FBQyxvQkFBb0IsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUNwSSxJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsSUFBSSxJQUFJLENBQUMsYUFBYSxFQUFFO1lBQ2xELE1BQU0sSUFBSSxLQUFLLENBQUMsMkNBQTJDLElBQUksQ0FBQyxPQUFPLHdDQUF3QyxDQUFDLENBQUM7U0FDbEg7UUFDRCxJQUFJLE9BQU8sQ0FBQyxnQkFBZ0IsRUFBRTtZQUM1QixTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7U0FDekM7UUFDRCxPQUFPLGFBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRSxJQUFJLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxXQUFXO1FBQ2hCLE9BQU87WUFDTCxJQUFJLEVBQUUsc0JBQVMsQ0FBQyxNQUFNO1lBQ3RCLE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztZQUNyQixHQUFHLElBQUksQ0FBQyxpQkFBaUIsRUFBRTtZQUMzQixHQUFHLElBQUksQ0FBQyxhQUFhLEVBQUU7U0FDeEIsQ0FBQztJQUNKLENBQUM7Q0FDRjtBQXJERCx3QkFxREM7QUEyQkQ7O0dBRUc7QUFDSCxNQUFNLGFBQWE7SUFDakIsWUFBNkIsTUFBYztRQUFkLFdBQU0sR0FBTixNQUFNLENBQVE7SUFDM0MsQ0FBQztJQUVNLElBQUksQ0FBQyxLQUFpQjtRQUMzQixJQUFJLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM3QixPQUFPLGFBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLE1BQU0sRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM1QyxDQUFDO0NBQ0YiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDaGFpbiB9IGZyb20gJy4uL2NoYWluJztcbmltcG9ydCB7IENvbmRpdGlvbiB9IGZyb20gJy4uL2NvbmRpdGlvbic7XG5pbXBvcnQgeyBJQ2hhaW5hYmxlLCBJTmV4dGFibGUgfSBmcm9tICcuLi90eXBlcyc7XG5pbXBvcnQgeyBTdGF0ZVR5cGUgfSBmcm9tICcuL3ByaXZhdGUvc3RhdGUtdHlwZSc7XG5pbXBvcnQgeyBTdGF0ZSB9IGZyb20gJy4vc3RhdGUnO1xuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGRlZmluaW5nIGEgQ2hvaWNlIHN0YXRlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ2hvaWNlUHJvcHMge1xuICAvKipcbiAgICogQW4gb3B0aW9uYWwgZGVzY3JpcHRpb24gZm9yIHRoaXMgc3RhdGVcbiAgICpcbiAgICogQGRlZmF1bHQgTm8gY29tbWVudFxuICAgKi9cbiAgcmVhZG9ubHkgY29tbWVudD86IHN0cmluZztcblxuICAvKipcbiAgICogSlNPTlBhdGggZXhwcmVzc2lvbiB0byBzZWxlY3QgcGFydCBvZiB0aGUgc3RhdGUgdG8gYmUgdGhlIGlucHV0IHRvIHRoaXMgc3RhdGUuXG4gICAqXG4gICAqIE1heSBhbHNvIGJlIHRoZSBzcGVjaWFsIHZhbHVlIERJU0NBUkQsIHdoaWNoIHdpbGwgY2F1c2UgdGhlIGVmZmVjdGl2ZVxuICAgKiBpbnB1dCB0byBiZSB0aGUgZW1wdHkgb2JqZWN0IHt9LlxuICAgKlxuICAgKiBAZGVmYXVsdCAkXG4gICAqL1xuICByZWFkb25seSBpbnB1dFBhdGg/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEpTT05QYXRoIGV4cHJlc3Npb24gdG8gc2VsZWN0IHBhcnQgb2YgdGhlIHN0YXRlIHRvIGJlIHRoZSBvdXRwdXQgdG8gdGhpcyBzdGF0ZS5cbiAgICpcbiAgICogTWF5IGFsc28gYmUgdGhlIHNwZWNpYWwgdmFsdWUgRElTQ0FSRCwgd2hpY2ggd2lsbCBjYXVzZSB0aGUgZWZmZWN0aXZlXG4gICAqIG91dHB1dCB0byBiZSB0aGUgZW1wdHkgb2JqZWN0IHt9LlxuICAgKlxuICAgKiBAZGVmYXVsdCAkXG4gICAqL1xuICByZWFkb25seSBvdXRwdXRQYXRoPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIERlZmluZSBhIENob2ljZSBpbiB0aGUgc3RhdGUgbWFjaGluZVxuICpcbiAqIEEgY2hvaWNlIHN0YXRlIGNhbiBiZSB1c2VkIHRvIG1ha2UgZGVjaXNpb25zIGJhc2VkIG9uIHRoZSBleGVjdXRpb25cbiAqIHN0YXRlLlxuICovXG5leHBvcnQgY2xhc3MgQ2hvaWNlIGV4dGVuZHMgU3RhdGUge1xuICBwdWJsaWMgcmVhZG9ubHkgZW5kU3RhdGVzOiBJTmV4dGFibGVbXSA9IFtdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2hvaWNlUHJvcHMgPSB7fSkge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICB9XG5cbiAgLyoqXG4gICAqIElmIHRoZSBnaXZlbiBjb25kaXRpb24gbWF0Y2hlcywgY29udGludWUgZXhlY3V0aW9uIHdpdGggdGhlIGdpdmVuIHN0YXRlXG4gICAqL1xuICBwdWJsaWMgd2hlbihjb25kaXRpb246IENvbmRpdGlvbiwgbmV4dDogSUNoYWluYWJsZSk6IENob2ljZSB7XG4gICAgc3VwZXIuYWRkQ2hvaWNlKGNvbmRpdGlvbiwgbmV4dC5zdGFydFN0YXRlKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBJZiBub25lIG9mIHRoZSBnaXZlbiBjb25kaXRpb25zIG1hdGNoLCBjb250aW51ZSBleGVjdXRpb24gd2l0aCB0aGUgZ2l2ZW4gc3RhdGVcbiAgICpcbiAgICogSWYgbm8gY29uZGl0aW9ucyBtYXRjaCBhbmQgbm8gb3RoZXJ3aXNlKCkgaGFzIGJlZW4gZ2l2ZW4sIGFuIGV4ZWN1dGlvblxuICAgKiBlcnJvciB3aWxsIGJlIHJhaXNlZC5cbiAgICovXG4gIHB1YmxpYyBvdGhlcndpc2UoZGVmOiBJQ2hhaW5hYmxlKTogQ2hvaWNlIHtcbiAgICBzdXBlci5tYWtlRGVmYXVsdChkZWYuc3RhcnRTdGF0ZSk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIGEgQ2hhaW4gdGhhdCBjb250YWlucyBhbGwgcmVhY2hhYmxlIGVuZCBzdGF0ZXMgZnJvbSB0aGlzIENob2ljZVxuICAgKlxuICAgKiBVc2UgdGhpcyB0byBjb21iaW5lIGFsbCBwb3NzaWJsZSBjaG9pY2UgcGF0aHMgYmFjay5cbiAgICovXG4gIHB1YmxpYyBhZnRlcndhcmRzKG9wdGlvbnM6IEFmdGVyd2FyZHNPcHRpb25zID0ge30pOiBDaGFpbiB7XG4gICAgY29uc3QgZW5kU3RhdGVzID0gU3RhdGUuZmlsdGVyTmV4dGFibGVzKFN0YXRlLmZpbmRSZWFjaGFibGVFbmRTdGF0ZXModGhpcywgeyBpbmNsdWRlRXJyb3JIYW5kbGVyczogb3B0aW9ucy5pbmNsdWRlRXJyb3JIYW5kbGVycyB9KSk7XG4gICAgaWYgKG9wdGlvbnMuaW5jbHVkZU90aGVyd2lzZSAmJiB0aGlzLmRlZmF1bHRDaG9pY2UpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJ2luY2x1ZGVPdGhlcndpc2UnIHNldCBidXQgQ2hvaWNlIHN0YXRlICR7dGhpcy5zdGF0ZUlkfSBhbHJlYWR5IGhhcyBhbiAnb3RoZXJ3aXNlJyB0cmFuc2l0aW9uYCk7XG4gICAgfVxuICAgIGlmIChvcHRpb25zLmluY2x1ZGVPdGhlcndpc2UpIHtcbiAgICAgIGVuZFN0YXRlcy5wdXNoKG5ldyBEZWZhdWx0QXNOZXh0KHRoaXMpKTtcbiAgICB9XG4gICAgcmV0dXJuIENoYWluLmN1c3RvbSh0aGlzLCBlbmRTdGF0ZXMsIHRoaXMpO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybiB0aGUgQW1hem9uIFN0YXRlcyBMYW5ndWFnZSBvYmplY3QgZm9yIHRoaXMgc3RhdGVcbiAgICovXG4gIHB1YmxpYyB0b1N0YXRlSnNvbigpOiBvYmplY3Qge1xuICAgIHJldHVybiB7XG4gICAgICBUeXBlOiBTdGF0ZVR5cGUuQ0hPSUNFLFxuICAgICAgQ29tbWVudDogdGhpcy5jb21tZW50LFxuICAgICAgLi4udGhpcy5yZW5kZXJJbnB1dE91dHB1dCgpLFxuICAgICAgLi4udGhpcy5yZW5kZXJDaG9pY2VzKCksXG4gICAgfTtcbiAgfVxufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIHNlbGVjdGluZyB0aGUgY2hvaWNlIHBhdGhzXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWZ0ZXJ3YXJkc09wdGlvbnMge1xuICAvKipcbiAgICogV2hldGhlciB0byBpbmNsdWRlIGVycm9yIGhhbmRsaW5nIHN0YXRlc1xuICAgKlxuICAgKiBJZiB0aGlzIGlzIHRydWUsIGFsbCBzdGF0ZXMgd2hpY2ggYXJlIGVycm9yIGhhbmRsZXJzIChhZGRlZCB0aHJvdWdoICdvbkVycm9yJylcbiAgICogYW5kIHN0YXRlcyByZWFjaGFibGUgdmlhIGVycm9yIGhhbmRsZXJzIHdpbGwgYmUgaW5jbHVkZWQgYXMgd2VsbC5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IGluY2x1ZGVFcnJvckhhbmRsZXJzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogV2hldGhlciB0byBpbmNsdWRlIHRoZSBkZWZhdWx0L290aGVyd2lzZSB0cmFuc2l0aW9uIGZvciB0aGUgY3VycmVudCBDaG9pY2Ugc3RhdGVcbiAgICpcbiAgICogSWYgdGhpcyBpcyB0cnVlIGFuZCB0aGUgY3VycmVudCBDaG9pY2UgZG9lcyBub3QgaGF2ZSBhIGRlZmF1bHQgb3V0Z29pbmdcbiAgICogdHJhbnNpdGlvbiwgb25lIHdpbGwgYmUgYWRkZWQgaW5jbHVkZWQgd2hlbiAubmV4dCgpIGlzIGNhbGxlZCBvbiB0aGUgY2hhaW4uXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBpbmNsdWRlT3RoZXJ3aXNlPzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBBZGFwdGVyIHRvIG1ha2UgdGhlIC5vdGhlcndpc2UoKSB0cmFuc2l0aW9uIHNldHRhYmxlIHRocm91Z2ggLm5leHQoKVxuICovXG5jbGFzcyBEZWZhdWx0QXNOZXh0IGltcGxlbWVudHMgSU5leHRhYmxlIHtcbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBjaG9pY2U6IENob2ljZSkge1xuICB9XG5cbiAgcHVibGljIG5leHQoc3RhdGU6IElDaGFpbmFibGUpOiBDaGFpbiB7XG4gICAgdGhpcy5jaG9pY2Uub3RoZXJ3aXNlKHN0YXRlKTtcbiAgICByZXR1cm4gQ2hhaW4uc2VxdWVuY2UodGhpcy5jaG9pY2UsIHN0YXRlKTtcbiAgfVxufVxuIl19