"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachine = exports.LogLevel = exports.StateMachineType = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const state_graph_1 = require("./state-graph");
const stepfunctions_generated_1 = require("./stepfunctions.generated");
/**
 * Two types of state machines are available in AWS Step Functions: EXPRESS AND STANDARD.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-standard-vs-express.html
 *
 * @default STANDARD
 */
var StateMachineType;
(function (StateMachineType) {
    /**
     * Express Workflows are ideal for high-volume, event processing workloads.
     */
    StateMachineType["EXPRESS"] = "EXPRESS";
    /**
     * Standard Workflows are ideal for long-running, durable, and auditable workflows.
     */
    StateMachineType["STANDARD"] = "STANDARD";
})(StateMachineType = exports.StateMachineType || (exports.StateMachineType = {}));
/**
 * Defines which category of execution history events are logged.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html
 *
 * @default ERROR
 */
var LogLevel;
(function (LogLevel) {
    /**
     * No Logging
     */
    LogLevel["OFF"] = "OFF";
    /**
     * Log everything
     */
    LogLevel["ALL"] = "ALL";
    /**
     * Log all errors
     */
    LogLevel["ERROR"] = "ERROR";
    /**
     * Log fatal errors
     */
    LogLevel["FATAL"] = "FATAL";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
/**
 * A new or imported state machine.
 */
class StateMachineBase extends core_1.Resource {
    /**
     * Import a state machine
     */
    static fromStateMachineArn(scope, id, stateMachineArn) {
        class Import extends StateMachineBase {
            constructor() {
                super(...arguments);
                this.stateMachineArn = stateMachineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant the given identity permissions to start an execution of this state
     * machine.
     */
    grantStartExecution(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions to read results from state
     * machine.
     */
    grantRead(identity) {
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListExecutions',
                'states:ListStateMachines',
            ],
            resourceArns: [this.stateMachineArn],
        });
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:DescribeExecution',
                'states:DescribeStateMachineForExecution',
                'states:GetExecutionHistory',
            ],
            resourceArns: [`${this.executionArn()}:*`],
        });
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListActivities',
                'states:DescribeStateMachine',
                'states:DescribeActivity',
            ],
            resourceArns: ['*'],
        });
    }
    /**
     * Grant the given identity task response permissions on a state machine
     */
    grantTaskResponse(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:SendTaskSuccess',
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
            ],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions on all executions of the state machine
     */
    grantExecution(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [`${this.executionArn()}:*`],
        });
    }
    /**
     * Grant the given identity custom permissions
     */
    grant(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Returns the pattern for the execution ARN's of the state machine
     */
    executionArn() {
        return core_1.Stack.of(this).formatArn({
            resource: 'execution',
            service: 'states',
            resourceName: core_1.Arn.parse(this.stateMachineArn, ':').resourceName,
            sep: ':',
        });
    }
}
/**
 * Define a StepFunctions State Machine
 */
class StateMachine extends StateMachineBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.stateMachineName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        });
        const graph = new state_graph_1.StateGraph(props.definition.startState, `State Machine ${id} definition`);
        graph.timeout = props.timeout;
        this.stateMachineType = props.stateMachineType ? props.stateMachineType : StateMachineType.STANDARD;
        const resource = new stepfunctions_generated_1.CfnStateMachine(this, 'Resource', {
            stateMachineName: this.physicalName,
            stateMachineType: props.stateMachineType ? props.stateMachineType : undefined,
            roleArn: this.role.roleArn,
            definitionString: core_1.Stack.of(this).toJsonString(graph.toGraphJson()),
            loggingConfiguration: props.logs ? this.buildLoggingConfiguration(props.logs) : undefined,
            tracingConfiguration: props.tracingEnabled ? this.buildTracingConfiguration() : undefined,
        });
        resource.node.addDependency(this.role);
        for (const statement of graph.policyStatements) {
            this.addToRolePolicy(statement);
        }
        this.stateMachineName = this.getResourceNameAttribute(resource.attrName);
        this.stateMachineArn = this.getResourceArnAttribute(resource.ref, {
            service: 'states',
            resource: 'stateMachine',
            resourceName: this.physicalName,
            sep: ':',
        });
    }
    /**
     * Add the given statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Return the given named metric for this State Machine's executions
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { StateMachineArn: this.stateMachineArn },
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of executions that failed
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.metric('ExecutionsFailed', props);
    }
    /**
     * Metric for the number of executions that were throttled
     *
     * @default sum over 5 minutes
     */
    metricThrottled(props) {
        return this.metric('ExecutionThrottled', props);
    }
    /**
     * Metric for the number of executions that were aborted
     *
     * @default sum over 5 minutes
     */
    metricAborted(props) {
        return this.metric('ExecutionsAborted', props);
    }
    /**
     * Metric for the number of executions that succeeded
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.metric('ExecutionsSucceeded', props);
    }
    /**
     * Metric for the number of executions that succeeded
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.metric('ExecutionsTimedOut', props);
    }
    /**
     * Metric for the number of executions that were started
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.metric('ExecutionsStarted', props);
    }
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes
     *
     * @default sum over 5 minutes
     */
    metricTime(props) {
        return this.metric('ExecutionTime', props);
    }
    buildLoggingConfiguration(logOptions) {
        // https://docs.aws.amazon.com/step-functions/latest/dg/cw-logs.html#cloudwatch-iam-policy
        this.addToRolePolicy(new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: [
                'logs:CreateLogDelivery',
                'logs:GetLogDelivery',
                'logs:UpdateLogDelivery',
                'logs:DeleteLogDelivery',
                'logs:ListLogDeliveries',
                'logs:PutResourcePolicy',
                'logs:DescribeResourcePolicies',
                'logs:DescribeLogGroups',
            ],
            resources: ['*'],
        }));
        return {
            destinations: [{
                    cloudWatchLogsLogGroup: { logGroupArn: logOptions.destination.logGroupArn },
                }],
            includeExecutionData: logOptions.includeExecutionData,
            level: logOptions.level || 'ERROR',
        };
    }
    buildTracingConfiguration() {
        this.addToRolePolicy(new iam.PolicyStatement({
            // https://docs.aws.amazon.com/xray/latest/devguide/security_iam_id-based-policy-examples.html#xray-permissions-resources
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            enabled: true,
        };
    }
}
exports.StateMachine = StateMachine;
//# sourceMappingURL=data:application/json;base64,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