import os.path

import six

from pprofile_ext import html


def sorted_files(pdict):
    """
    Returns the file_summary information sorted in reverse order based on duration

    :param pdict: profile dict

    :return: sorted list of file_summary dicts
    """
    return sorted([v['file_summary'] for k, v in six.iteritems(pdict) if k != 'summary'],
                  key=lambda f: f['duration'], reverse=True)


def html_summary(pdict, output_dir):
    """
    Generate the HTML for the summary page displaying the files included
    in the profile in descending order of file execution time.

    :param pdict: dictionary holding the parsed profile information
    :param output_dir: location where summary file is written.

    :return: dictionary holding the parsed profile information, including any
             updates added by html_summary
    """
    file_summaries = sorted_files(pdict)
    total_time = sum([f['duration'] for f in file_summaries])

    # column_specs defines each column of the table generated by html_summary
    column_specs = (html.column_spec('',
                                     lambda f: html.box(0., f['duration'], total_time),
                                     50,
                                     padding_left=0,
                                     padding_right=0),
                    html.column_spec('time in sec',
                                     lambda f: '{0:.4f}'.format(f['duration']),
                                     100),
                    html.column_spec('%',
                                     lambda f: '{0:.4f}'.format(f['percentage']),
                                     100),
                    html.column_spec('filename',
                                     lambda f: html.href(html.get_html_filename('', f['name']),
                                                         html.strip_pointy(f['name'])),
                                     None,
                                     padding_right=0,
                                     align='left')
                    )

    # generate the html
    h = html.html()

    with h.preamble():
        with h.title():
            h.add('most expensive files')
        with h.table():
            # header row
            with h.table_row():
                for c in column_specs:
                    h.add(c(None, header=True))
            # content rows
            for fs in file_summaries:
                with h.table_row():
                    for c in column_specs:
                        h.add(c(fs))

    # write the file to disk
    with open(os.path.abspath(os.path.join(output_dir, 'index.html')), 'w') as f:
        f.write(''.join(h.buffer))

    # return the parsed dictionary
    return pdict
