"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Table = exports.TableEncryption = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const glue_generated_1 = require("./glue.generated");
/**
 * Encryption options for a Table.
 *
 * @see https://docs.aws.amazon.com/athena/latest/ug/encryption.html
 */
var TableEncryption;
(function (TableEncryption) {
    TableEncryption["UNENCRYPTED"] = "Unencrypted";
    /**
     * Server side encryption (SSE) with an Amazon S3-managed key.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingServerSideEncryption.html
     */
    TableEncryption["S3_MANAGED"] = "SSE-S3";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingKMSEncryption.html
     */
    TableEncryption["KMS"] = "SSE-KMS";
    /**
     * Server-side encryption (SSE) with an AWS KMS key managed by the KMS service.
     */
    TableEncryption["KMS_MANAGED"] = "SSE-KMS-MANAGED";
    /**
     * Client-side encryption (CSE) with an AWS KMS key managed by the account owner.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingClientSideEncryption.html
     */
    TableEncryption["CLIENT_SIDE_KMS"] = "CSE-KMS";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
/**
 * A Glue table.
 */
class Table extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.tableName,
        });
        /**
         * Partition indexes must be created one at a time. To avoid
         * race conditions, we store the resource and add dependencies
         * each time a new partition index is created.
         */
        this.partitionIndexCustomResources = [];
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Table);
            }
            throw error;
        }
        this.database = props.database;
        this.dataFormat = props.dataFormat;
        this.s3Prefix = props.s3Prefix ?? '';
        validateSchema(props.columns, props.partitionKeys);
        this.columns = props.columns;
        this.partitionKeys = props.partitionKeys;
        this.compressed = props.compressed ?? false;
        const { bucket, encryption, encryptionKey } = createBucket(this, props);
        this.bucket = bucket;
        this.encryption = encryption;
        this.encryptionKey = encryptionKey;
        const tableResource = new glue_generated_1.CfnTable(this, 'Table', {
            catalogId: props.database.catalogId,
            databaseName: props.database.databaseName,
            tableInput: {
                name: this.physicalName,
                description: props.description || `${props.tableName} generated by CDK`,
                partitionKeys: renderColumns(props.partitionKeys),
                parameters: {
                    classification: props.dataFormat.classificationString?.value,
                    has_encrypted_data: this.encryption !== TableEncryption.UNENCRYPTED,
                },
                storageDescriptor: {
                    location: `s3://${this.bucket.bucketName}/${this.s3Prefix}`,
                    compressed: this.compressed,
                    storedAsSubDirectories: props.storedAsSubDirectories ?? false,
                    columns: renderColumns(props.columns),
                    inputFormat: props.dataFormat.inputFormat.className,
                    outputFormat: props.dataFormat.outputFormat.className,
                    serdeInfo: {
                        serializationLibrary: props.dataFormat.serializationLibrary.className,
                    },
                },
                tableType: 'EXTERNAL_TABLE',
            },
        });
        this.tableName = this.getResourceNameAttribute(tableResource.ref);
        this.tableArn = this.stack.formatArn({
            service: 'glue',
            resource: 'table',
            resourceName: `${this.database.databaseName}/${this.tableName}`,
        });
        this.node.defaultChild = tableResource;
        // Partition index creation relies on created table.
        if (props.partitionIndexes) {
            this.partitionIndexes = props.partitionIndexes;
            this.partitionIndexes.forEach((index) => this.addPartitionIndex(index));
        }
    }
    static fromTableArn(scope, id, tableArn) {
        const tableName = core_1.Fn.select(1, core_1.Fn.split('/', core_1.Stack.of(scope).splitArn(tableArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName));
        return Table.fromTableAttributes(scope, id, {
            tableArn,
            tableName,
        });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.tableArn = attrs.tableArn;
                this.tableName = attrs.tableName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a partition index to the table. You can have a maximum of 3 partition
     * indexes to a table. Partition index keys must be a subset of the table's
     * partition keys.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/partition-indexes.html
     */
    addPartitionIndex(index) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_PartitionIndex(index);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addPartitionIndex);
            }
            throw error;
        }
        const numPartitions = this.partitionIndexCustomResources.length;
        if (numPartitions >= 3) {
            throw new Error('Maximum number of partition indexes allowed is 3');
        }
        this.validatePartitionIndex(index);
        const indexName = index.indexName ?? this.generateIndexName(index.keyNames);
        const partitionIndexCustomResource = new cr.AwsCustomResource(this, `partition-index-${indexName}`, {
            onCreate: {
                service: 'Glue',
                action: 'createPartitionIndex',
                parameters: {
                    DatabaseName: this.database.databaseName,
                    TableName: this.tableName,
                    PartitionIndex: {
                        IndexName: indexName,
                        Keys: index.keyNames,
                    },
                },
                physicalResourceId: cr.PhysicalResourceId.of(indexName),
            },
            policy: cr.AwsCustomResourcePolicy.fromSdkCalls({
                resources: cr.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        this.grantToUnderlyingResources(partitionIndexCustomResource, ['glue:UpdateTable']);
        // Depend on previous partition index if possible, to avoid race condition
        if (numPartitions > 0) {
            this.partitionIndexCustomResources[numPartitions - 1].node.addDependency(partitionIndexCustomResource);
        }
        this.partitionIndexCustomResources.push(partitionIndexCustomResource);
    }
    generateIndexName(keys) {
        const prefix = keys.join('-') + '-';
        const uniqueId = core_1.Names.uniqueId(this);
        const maxIndexLength = 80; // arbitrarily specified
        const startIndex = Math.max(0, uniqueId.length - (maxIndexLength - prefix.length));
        return prefix + uniqueId.substring(startIndex);
    }
    validatePartitionIndex(index) {
        if (index.indexName !== undefined && (index.indexName.length < 1 || index.indexName.length > 255)) {
            throw new Error(`Index name must be between 1 and 255 characters, but got ${index.indexName.length}`);
        }
        if (!this.partitionKeys || this.partitionKeys.length === 0) {
            throw new Error('The table must have partition keys to create a partition index');
        }
        const keyNames = this.partitionKeys.map(pk => pk.name);
        if (!index.keyNames.every(k => keyNames.includes(k))) {
            throw new Error(`All index keys must also be partition keys. Got ${index.keyNames} but partition key names are ${keyNames}`);
        }
    }
    /**
     * Grant read permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantRead(grantee) {
        const ret = this.grant(grantee, readPermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantDecrypt(grantee);
        }
        this.bucket.grantRead(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantWrite(grantee) {
        const ret = this.grant(grantee, writePermissions);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncrypt(grantee);
        }
        this.bucket.grantWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant read and write permissions to the table and the underlying data stored in S3 to an IAM principal.
     *
     * @param grantee the principal
     */
    grantReadWrite(grantee) {
        const ret = this.grant(grantee, [...readPermissions, ...writePermissions]);
        if (this.encryptionKey && this.encryption === TableEncryption.CLIENT_SIDE_KMS) {
            this.encryptionKey.grantEncryptDecrypt(grantee);
        }
        this.bucket.grantReadWrite(grantee, this.getS3PrefixForGrant());
        return ret;
    }
    /**
     * Grant the given identity custom permissions.
     */
    grant(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [this.tableArn],
            actions,
        });
    }
    /**
     * Grant the given identity custom permissions to ALL underlying resources of the table.
     * Permissions will be granted to the catalog, the database, and the table.
     */
    grantToUnderlyingResources(grantee, actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            resourceArns: [
                this.tableArn,
                this.database.catalogArn,
                this.database.databaseArn,
            ],
            actions,
        });
    }
    getS3PrefixForGrant() {
        return this.s3Prefix + '*';
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-glue.Table", version: "1.179.0" };
function validateSchema(columns, partitionKeys) {
    if (columns.length === 0) {
        throw new Error('you must specify at least one column for the table');
    }
    // Check there is at least one column and no duplicated column names or partition keys.
    const names = new Set();
    (columns.concat(partitionKeys || [])).forEach(column => {
        if (names.has(column.name)) {
            throw new Error(`column names and partition keys must be unique, but \'${column.name}\' is duplicated`);
        }
        names.add(column.name);
    });
}
// map TableEncryption to bucket's SSE configuration (s3.BucketEncryption)
const encryptionMappings = {
    [TableEncryption.S3_MANAGED]: s3.BucketEncryption.S3_MANAGED,
    [TableEncryption.KMS_MANAGED]: s3.BucketEncryption.KMS_MANAGED,
    [TableEncryption.KMS]: s3.BucketEncryption.KMS,
    [TableEncryption.CLIENT_SIDE_KMS]: s3.BucketEncryption.UNENCRYPTED,
    [TableEncryption.UNENCRYPTED]: s3.BucketEncryption.UNENCRYPTED,
};
// create the bucket to store a table's data depending on the `encryption` and `encryptionKey` properties.
function createBucket(table, props) {
    const encryption = props.encryption || TableEncryption.UNENCRYPTED;
    let bucket = props.bucket;
    if (bucket && (encryption !== TableEncryption.UNENCRYPTED && encryption !== TableEncryption.CLIENT_SIDE_KMS)) {
        throw new Error('you can not specify encryption settings if you also provide a bucket');
    }
    let encryptionKey;
    if (encryption === TableEncryption.CLIENT_SIDE_KMS && props.encryptionKey === undefined) {
        // CSE-KMS should behave the same as SSE-KMS - use the provided key or create one automatically
        // Since Bucket only knows about SSE, we repeat the logic for CSE-KMS at the Table level.
        encryptionKey = new kms.Key(table, 'Key');
    }
    else {
        encryptionKey = props.encryptionKey;
    }
    // create the bucket if none was provided
    if (!bucket) {
        if (encryption === TableEncryption.CLIENT_SIDE_KMS) {
            bucket = new s3.Bucket(table, 'Bucket');
        }
        else {
            bucket = new s3.Bucket(table, 'Bucket', {
                encryption: encryptionMappings[encryption],
                encryptionKey,
            });
            encryptionKey = bucket.encryptionKey;
        }
    }
    return {
        bucket,
        encryption,
        encryptionKey,
    };
}
const readPermissions = [
    'glue:BatchGetPartition',
    'glue:GetPartition',
    'glue:GetPartitions',
    'glue:GetTable',
    'glue:GetTables',
    'glue:GetTableVersion',
    'glue:GetTableVersions',
];
const writePermissions = [
    'glue:BatchCreatePartition',
    'glue:BatchDeletePartition',
    'glue:CreatePartition',
    'glue:DeletePartition',
    'glue:UpdatePartition',
];
function renderColumns(columns) {
    if (columns === undefined) {
        return undefined;
    }
    return columns.map(column => {
        return {
            name: column.name,
            type: column.type.inputString,
            comment: column.comment,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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