"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Job = exports.MetricType = exports.JobState = exports.WorkerType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const _1 = require(".");
const glue_generated_1 = require("./glue.generated");
/**
 * The type of predefined worker that is allocated when a job runs.
 *
 * If you need to use a WorkerType that doesn't exist as a static member, you
 * can instantiate a `WorkerType` object, e.g: `WorkerType.of('other type')`.
 */
class WorkerType {
    constructor(name) {
        this.name = name;
    }
    /**
     * Custom worker type
     * @param workerType custom worker type
     */
    static of(workerType) {
        return new WorkerType(workerType);
    }
}
exports.WorkerType = WorkerType;
_a = JSII_RTTI_SYMBOL_1;
WorkerType[_a] = { fqn: "@aws-cdk/aws-glue.WorkerType", version: "1.179.0" };
/**
 * Each worker provides 4 vCPU, 16 GB of memory and a 50GB disk, and 2 executors per worker.
 */
WorkerType.STANDARD = new WorkerType('Standard');
/**
 * Each worker maps to 1 DPU (4 vCPU, 16 GB of memory, 64 GB disk), and provides 1 executor per worker. Suitable for memory-intensive jobs.
 */
WorkerType.G_1X = new WorkerType('G.1X');
/**
 * Each worker maps to 2 DPU (8 vCPU, 32 GB of memory, 128 GB disk), and provides 1 executor per worker. Suitable for memory-intensive jobs.
 */
WorkerType.G_2X = new WorkerType('G.2X');
/**
 * Job states emitted by Glue to CloudWatch Events.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types for more information.
 */
var JobState;
(function (JobState) {
    /**
     * State indicating job run succeeded
     */
    JobState["SUCCEEDED"] = "SUCCEEDED";
    /**
     * State indicating job run failed
     */
    JobState["FAILED"] = "FAILED";
    /**
     * State indicating job run timed out
     */
    JobState["TIMEOUT"] = "TIMEOUT";
    /**
     * State indicating job is starting
     */
    JobState["STARTING"] = "STARTING";
    /**
     * State indicating job is running
     */
    JobState["RUNNING"] = "RUNNING";
    /**
     * State indicating job is stopping
     */
    JobState["STOPPING"] = "STOPPING";
    /**
     * State indicating job stopped
     */
    JobState["STOPPED"] = "STOPPED";
})(JobState = exports.JobState || (exports.JobState = {}));
/**
 * The Glue CloudWatch metric type.
 *
 * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
 */
var MetricType;
(function (MetricType) {
    /**
     * A value at a point in time.
     */
    MetricType["GAUGE"] = "gauge";
    /**
     * An aggregate number.
     */
    MetricType["COUNT"] = "count";
})(MetricType = exports.MetricType || (exports.MetricType = {}));
class JobBase extends cdk.Resource {
    /**
     * Create a CloudWatch Event Rule for this Glue Job when it's in a given state
     *
     * @param id construct id
     * @param options event options. Note that some values are overridden if provided, these are
     *  - eventPattern.source = ['aws.glue']
     *  - eventPattern.detailType = ['Glue Job State Change', 'Glue Job Run Status']
     *  - eventPattern.detail.jobName = [this.jobName]
     *
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.glue'],
            detailType: ['Glue Job State Change', 'Glue Job Run Status'],
            detail: {
                jobName: [this.jobName],
            },
        });
        return rule;
    }
    /**
     * Create a CloudWatch Event Rule for the transition into the input jobState.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @param options optional event options.
     */
    onStateChange(id, jobState, options = {}) {
        const rule = this.onEvent(id, {
            description: `Rule triggered when Glue job ${this.jobName} is in ${jobState} state`,
            ...options,
        });
        rule.addEventPattern({
            detail: {
                state: [jobState],
            },
        });
        return rule;
    }
    /**
     * Create a CloudWatch Event Rule matching JobState.SUCCEEDED.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onSuccess(id, options = {}) {
        return this.onStateChange(id, JobState.SUCCEEDED, options);
    }
    /**
     * Return a CloudWatch Event Rule matching FAILED state.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onFailure(id, options = {}) {
        return this.onStateChange(id, JobState.FAILED, options);
    }
    /**
     * Return a CloudWatch Event Rule matching TIMEOUT state.
     *
     * @param id construct id.
     * @param options optional event options. default is {}.
     */
    onTimeout(id, options = {}) {
        return this.onStateChange(id, JobState.TIMEOUT, options);
    }
    /**
     * Create a CloudWatch metric.
     *
     * @param metricName name of the metric typically prefixed with `glue.driver.`, `glue.<executorId>.` or `glue.ALL.`.
     * @param type the metric type.
     * @param props metric options.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/monitoring-awsglue-with-cloudwatch-metrics.html
     */
    metric(metricName, type, props) {
        return new cloudwatch.Metric({
            metricName,
            namespace: 'Glue',
            dimensionsMap: {
                JobName: this.jobName,
                JobRunId: 'ALL',
                Type: type,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Return a CloudWatch Metric indicating job success.
     *
     * This metric is based on the Rule returned by no-args onSuccess() call.
     */
    metricSuccess(props) {
        return metricRule(this.metricJobStateRule('SuccessMetricRule', JobState.SUCCEEDED), props);
    }
    /**
     * Return a CloudWatch Metric indicating job failure.
     *
     * This metric is based on the Rule returned by no-args onFailure() call.
     */
    metricFailure(props) {
        return metricRule(this.metricJobStateRule('FailureMetricRule', JobState.FAILED), props);
    }
    /**
     * Return a CloudWatch Metric indicating job timeout.
     *
     * This metric is based on the Rule returned by no-args onTimeout() call.
     */
    metricTimeout(props) {
        return metricRule(this.metricJobStateRule('TimeoutMetricRule', JobState.TIMEOUT), props);
    }
    /**
     * Creates or retrieves a singleton event rule for the input job state for use with the metric JobState methods.
     *
     * @param id construct id.
     * @param jobState the job state.
     * @private
     */
    metricJobStateRule(id, jobState) {
        return this.node.tryFindChild(id) ?? this.onStateChange(id, jobState);
    }
}
/**
 * A Glue Job.
 */
class Job extends JobBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_JobProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Job);
            }
            throw error;
        }
        const executable = props.executable.bind();
        this.role = props.role ?? new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('glue.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole')],
        });
        this.grantPrincipal = this.role;
        const sparkUI = props.sparkUI?.enabled ? this.setupSparkUI(executable, this.role, props.sparkUI) : undefined;
        this.sparkUILoggingLocation = sparkUI?.location;
        const continuousLoggingArgs = props.continuousLogging?.enabled ? this.setupContinuousLogging(this.role, props.continuousLogging) : {};
        const profilingMetricsArgs = props.enableProfilingMetrics ? { '--enable-metrics': '' } : {};
        const defaultArguments = {
            ...this.executableArguments(executable),
            ...continuousLoggingArgs,
            ...profilingMetricsArgs,
            ...sparkUI?.args,
            ...this.checkNoReservedArgs(props.defaultArguments),
        };
        const jobResource = new glue_generated_1.CfnJob(this, 'Resource', {
            name: props.jobName,
            description: props.description,
            role: this.role.roleArn,
            command: {
                name: executable.type.name,
                scriptLocation: this.codeS3ObjectUrl(executable.script),
                pythonVersion: executable.pythonVersion,
            },
            glueVersion: executable.glueVersion.name,
            workerType: props.workerType?.name,
            numberOfWorkers: props.workerCount,
            maxCapacity: props.maxCapacity,
            maxRetries: props.maxRetries,
            executionProperty: props.maxConcurrentRuns ? { maxConcurrentRuns: props.maxConcurrentRuns } : undefined,
            notificationProperty: props.notifyDelayAfter ? { notifyDelayAfter: props.notifyDelayAfter.toMinutes() } : undefined,
            timeout: props.timeout?.toMinutes(),
            connections: props.connections ? { connections: props.connections.map((connection) => connection.connectionName) } : undefined,
            securityConfiguration: props.securityConfiguration?.securityConfigurationName,
            tags: props.tags,
            defaultArguments,
        });
        const resourceName = this.getResourceNameAttribute(jobResource.ref);
        this.jobArn = jobArn(this, resourceName);
        this.jobName = resourceName;
    }
    /**
     * Creates a Glue Job
     *
     * @param scope The scope creating construct (usually `this`).
     * @param id The construct's id.
     * @param attrs Import attributes
     */
    static fromJobAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_glue_JobAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromJobAttributes);
            }
            throw error;
        }
        class Import extends JobBase {
            constructor() {
                super(...arguments);
                this.jobName = attrs.jobName;
                this.jobArn = jobArn(scope, attrs.jobName);
                this.grantPrincipal = attrs.role ?? new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Check no usage of reserved arguments.
     *
     * @see https://docs.aws.amazon.com/glue/latest/dg/aws-glue-programming-etl-glue-arguments.html
     */
    checkNoReservedArgs(defaultArguments) {
        if (defaultArguments) {
            const reservedArgs = new Set(['--conf', '--debug', '--mode', '--JOB_NAME']);
            Object.keys(defaultArguments).forEach((arg) => {
                if (reservedArgs.has(arg)) {
                    throw new Error(`The ${arg} argument is reserved by Glue. Don't set it`);
                }
            });
        }
        return defaultArguments;
    }
    executableArguments(config) {
        const args = {};
        args['--job-language'] = config.language;
        if (config.className) {
            args['--class'] = config.className;
        }
        if (config.extraJars && config.extraJars?.length > 0) {
            args['--extra-jars'] = config.extraJars.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraPythonFiles && config.extraPythonFiles.length > 0) {
            args['--extra-py-files'] = config.extraPythonFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraFiles && config.extraFiles.length > 0) {
            args['--extra-files'] = config.extraFiles.map(code => this.codeS3ObjectUrl(code)).join(',');
        }
        if (config.extraJarsFirst) {
            args['--user-jars-first'] = 'true';
        }
        return args;
    }
    setupSparkUI(executable, role, props) {
        if (_1.JobType.PYTHON_SHELL === executable.type) {
            throw new Error('Spark UI is not available for JobType.PYTHON_SHELL jobs');
        }
        const bucket = props.bucket ?? new s3.Bucket(this, 'SparkUIBucket');
        bucket.grantReadWrite(role);
        const args = {
            '--enable-spark-ui': 'true',
            '--spark-event-logs-path': bucket.s3UrlForObject(props.prefix),
        };
        return {
            location: {
                prefix: props.prefix,
                bucket,
            },
            args,
        };
    }
    setupContinuousLogging(role, props) {
        const args = {
            '--enable-continuous-cloudwatch-log': 'true',
            '--enable-continuous-log-filter': (props.quiet ?? true).toString(),
        };
        if (props.logGroup) {
            args['--continuous-log-logGroup'] = props.logGroup.logGroupName;
            props.logGroup.grantWrite(role);
        }
        if (props.logStreamPrefix) {
            args['--continuous-log-logStreamPrefix'] = props.logStreamPrefix;
        }
        if (props.conversionPattern) {
            args['--continuous-log-conversionPattern'] = props.conversionPattern;
        }
        return args;
    }
    codeS3ObjectUrl(code) {
        const s3Location = code.bind(this, this.role).s3Location;
        return `s3://${s3Location.bucketName}/${s3Location.objectKey}`;
    }
}
exports.Job = Job;
_b = JSII_RTTI_SYMBOL_1;
Job[_b] = { fqn: "@aws-cdk/aws-glue.Job", version: "1.179.0" };
/**
 * Create a CloudWatch Metric that's based on Glue Job events
 * {@see https://docs.aws.amazon.com/AmazonCloudWatch/latest/events/EventTypes.html#glue-event-types}
 * The metric has namespace = 'AWS/Events', metricName = 'TriggeredRules' and RuleName = rule.ruleName dimension.
 *
 * @param rule for use in setting RuleName dimension value
 * @param props metric properties
 */
function metricRule(rule, props) {
    return new cloudwatch.Metric({
        namespace: 'AWS/Events',
        metricName: 'TriggeredRules',
        dimensionsMap: { RuleName: rule.ruleName },
        statistic: cloudwatch.Statistic.SUM,
        ...props,
    }).attachTo(rule);
}
/**
 * Returns the job arn
 * @param scope
 * @param jobName
 */
function jobArn(scope, jobName) {
    return cdk.Stack.of(scope).formatArn({
        service: 'glue',
        resource: 'job',
        resourceName: jobName,
    });
}
//# sourceMappingURL=data:application/json;base64,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